﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief クライアントにおける消費型サービスアイテムの権利の型を定義します。
 */

#pragma once
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace ec {

/**
    @brief 一度に取り扱い可能な権利の最大数
*/
static const int ConsumableServiceItemRightDataCountMax = 20;

/**
    @brief ライブラリの内部処理に必要なワークバッファの最小メモリサイズ
*/
static const size_t RequiredWorkMemorySizeForConsumableServiceItem = 256 * 1024;

/**
    @brief 購入した消費型サービスアイテムを識別する ID です。最大 16 文字を文字列として保持します。

    @details
        ConsumableServiceItemRightData.IsServiceProvidableOnLocal() == true の場合に、この ID に紐づけたサービスをユーザーに提供してください。

        nn::ec::ConsumableServiceItemId は、消費型サービスアイテムへの各種リクエストを行うために必要なサーバーパラメータである、アイテム ID に相当します。
        アイテム ID はショップに登録する必要があります。
        登録方法や、ID の詳細は『Nintendo Switch サービスアイテム プログラミングマニュアル』の「開発を始めるための準備」を参照してください。
*/
struct ConsumableServiceItemId
{
    static const int Length = 16;
    char value[Length + 1];

    bool IsValid() const NN_NOEXCEPT;
};

//! 前方宣言
struct ConsumableServiceItemRightDataImpl;

/**
    @brief 消費型サービスアイテムの権利情報です。
*/
struct ConsumableServiceItemRightData
{
    nn::util::TypedStorage<ConsumableServiceItemRightDataImpl, 104, 8> value;

    bool IsValid() const NN_NOEXCEPT;

    /**
        @brief 権利情報から Item ID を取得します
    */
    ConsumableServiceItemId GetConsumableServiceItemId() const NN_NOEXCEPT;

    /**
        @brief 権利情報からサービスが提供可能になっているかを判定します。

        @details
            - IsServiceProvidableOnLocal() == true の場合

              任意のタイミングでサービスを提供してください。
              提供が完了したら、この権利情報を削除してください。

            - IsServiceProvidableOnLocal() == false の場合

              サーバー上で消費済みであることを確認できていないため、まだサービスを提供できません。
              消費済みかを確認するため、AsyncGetProvidableRightDataRequest に権利情報を渡して、サーバーに問い合わせてください。

              サービスの提供が可能だとサーバーで確認できた場合
              IsServiceProvidableOnLocal() == true を満たす権利情報が AsyncGetProvidableRightDataRequest の出力として返ってきます。
              逆に提供不可能な場合は、この権利情報は出力されません。

              サーバーへの問い合わせが成功した場合、もともと存在していた権利情報は古い情報になります。
              そのため、もともとの権利情報は、すぐに削除してください。
    */
    bool IsServiceProvidableOnLocal() const NN_NOEXCEPT;
};

/**
    @brief デシリアライズ時に、アプリケーションが確保する必要があるデータの情報です。

    @details
        シリアライズしたデータを GetSerializedConsumableServiceItemRightDataInfo に渡すことにより取得できます。
        デシリアライズ時に、rightDataCount の数だけ ConsumableServiceItemRightData のバッファを確保してください。
*/
struct SerializedConsumableServiceItemRightDataInfo
{
    int rightDataCount;
};

}}
