﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief ショップページを開くのに利用する API 宣言(内部開発者向け)
 */

#pragma once

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/applet/applet.h>
#include <nn/ec/detail/ec_ShowShopPageArg.h>
#include <nn/ec/detail/ec_ShowShopPageTypes.h>

namespace nn { namespace ec { namespace detail {

//------------------------------------------------------------------------
/**
 * @brief ショップページを開きます。
 *
 * @param[out]  pOutputShopPageReturnValue  ショップページを開いた戻り値の格納先を指定してください。
 * @param[in]   arg                         ショップページを開く際に利用されるパラメータです。
 */
nn::Result ShowShopPage(ShopPageReturnValue* pOutputShopPageReturnValue,
    const ShowShopPageArg& arg) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @brief 指定したストレージからショップページを開いた戻り値を取得します。
 *
 * @param[out]  pOutputShopPageReturnValue  ショップページを開いた戻り値の格納先を指定してください。
 * @param[in]   handle                      nn::applet::TryPopFromOutChannel で取得した nn::applet::StorageHandle を指定してください。
 *
 * @details Unwind 時に戻り値を取得する為の機能です。
 *          handle から値を取り出すだけでリリース等は行いません。
 */
nn::Result GetShopPageReturnValueFromStorage(ShopPageReturnValue* pOutputShopPageReturnValue,
    const nn::applet::StorageHandle& handle) NN_NOEXCEPT;

//------------------------------------------------------------------------
/**
 * @brief ショップページを開いた戻り値を設定するクラスです。
 */
class ShopPageReturnValueDetail
    : protected ShopPageReturnValue
{
public:
    /**
     * @brief 終了理由を設定します。
     *
     * @param[in]   shopExitReason 終了理由を指定してください。
     */
    inline void SetShopExitReason(ShopExitReason shopExitReason) NN_NOEXCEPT;

    /**
     * @brief 終了時開いていたページの URL を設定します。
     *
     * @param[in]   lastUrl 終了時開いていたページの URL を指定してください。
     *
     * @details 指定する値は NULL 終端されている必要があります。
     *          LastUrlMaxLength -1 文字以下にする必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    inline void SetLastUrl(const char* lastUrl) NN_NOEXCEPT;
};

//------------------------------------------------------------------------
void ShopPageReturnValueDetail::SetShopExitReason(ShopExitReason shopExitReason) NN_NOEXCEPT
{
    m_ExitReason = shopExitReason;
}

//------------------------------------------------------------------------
void ShopPageReturnValueDetail::SetLastUrl(const char* lastUrl) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(lastUrl);
    size_t size = std::strlen(lastUrl) + 1;
    NN_SDK_ASSERT(size <= LastUrlMaxLength);

    std::memcpy(m_LastUrl, lastUrl, size);

    m_LastUrlSize = uint64_t(size);
}

}}} // namespace nn::ec::detail
