﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a Record class provided
 * by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The Record class is used to parse DNS resource records.
 *
 * @details
 * The DNS Resource Record is the kind of record used with answer,
 * authority, and additional sections. For more information see
 * section 3.2 of RFC 1035.
 *
 * Please note that the Message object does not contain records,
 * rather it contains pointers to buffers that contain serialized
 * record bytes.
 */
class Record
{
private:
    /**
     * @brief A pointer to the Message that contains this object.
     */
    const Message* m_pMessage;

    /**
     * @brief This is the block of memory that refers to the buffer
     * that created the Record with Record::FromBuffer.
     */
    MemoryBlock m_Range;

    /**
     * @brief Indicates the object size or if it is uninitialized.
     */
    size_t m_DirtySize;

    /**
     * @brief The name of the DNS record.
     */
    Label m_Name;

    /**
     * @brief The type parameter corresponding to the record type
     * data. Values are found in @ref TypeConstant.
     */
    TypeConstant m_Type;

    /**
     * @brief The class of network that the record corresponds to.
     * Values are found in @ref ClassConstant.
     */
    ClassConstant m_Class;

    /**
     * @brief The Time-To-Live value.
     *
     * @details
     * According to The RFC (1035):
     *
     * "a 32 bit unsigned integer that specifies the time interval
     * (in seconds) that the resource record may be cached before it
     * should be discarded. Zero values are interpreted to mean that
     * the RR can only be used for the transaction in progress, and
     * should not be cached"
     */
    uint32_t m_TimeToLive;

    /**
     * @brief The resource record data length of the 'data' field.
     */
    uint16_t m_Length;

    /**
     * @brief Contains the DNS record data. This data is given
     * interpretive context by the @a m_Type and @a m_Class fields.
     */
    const uint8_t* m_pData;

public:

    /**
     * @brief Constructor.
     */
    Record();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The @a Record on the right hand side of the
     * expression.
     */
    Record(const Record& rhs);

    /**
     * @brief Destructor.
     */
    ~Record();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The @a Record on the right hand side of the
     * expression.
     */
    Record& operator=(const Record& rhs);

    /**
     * @brief Returns true if two @a Record objects are equal.
     *
     * @param[in] rhs The @a Record on the right hand side of the
     * expression.
     *
     * @return Returns true if both objects are content equivalent.
     */
    bool operator==(const Record& rhs) const NN_NOEXCEPT;

    /**
     * @brief Getter for @a m_pMessage.
     *
     * @return A reference to the message pointer.
     */
    const Message* & GetMessage() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_Name.
     *
     * @return A reference to the name label field.
     */
    Label& GetName() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_Type.
     *
     * @return A reference to the type field.
     */
    TypeConstant& GetType() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_Class.
     *
     * @return A reference to the class field.
     */
    ClassConstant& GetClass() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_TimeToLive.
     *
     * @return A reference to the time-to-live field.
     */
    uint32_t& GetTimeToLive() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for @a m_Length.
     *
     * @return A reference to the length field.
     */
    uint16_t& GetLength() NN_NOEXCEPT;

    /**
     * @brief Getter for @a m_Data.
     *
     * @return The data pointer.
     */
    uint8_t const *& GetData() NN_NOEXCEPT;

    /**
     * @brief Initializes the @a Record.
     *
     * @details
     * Initialize a @a Record with the provided
     * nn::dns::parser::Message: zeroes out all fields, sets dirty
     * flag to zero, and point the @a Record at the the provided
     * nn::dns::parser::Message.
     *
     * @param[in] pMessage The provided @a Message.
     */
    void Initialize(const Message* pMessage) NN_NOEXCEPT;

    /**
     * @brief Get the size of the @a Record for serialization.
     *
     * @return Returns error or success:
     *          - -1 : An error occurred.
     *          - >0 : The size of the object.
     */
    ssize_t SizeOf() const NN_NOEXCEPT;

    /**
     * @brief Reads a @a Record from the contents of a buffer in the
     * standard RFC format.
     *
     * @details It is the reverse of @ref Record::ToBuffer().
     *
     * @param[in] pBuffer The buffer that contains @a Record bytes.
     *
     * @param[in] size The size in bytes left in the buffer.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include a null buffer pointer or parsing error.
     */
    ssize_t FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief Writes a @a Record to a buffer in the RFC standard
     * format.
     *
     * @details it is the reverse of @ref Record::FromBuffer().
     *
     * @param[out] pOutBuffer The buffer that receives the @a Record
     * bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @return Returns the number of bytes written to the buffer or
     * -1 on error.
     *
     * Error cases include null buffer pointer, a buffer of
     * insufficient size, or a parsing error.
     */
    ssize_t ToBuffer(uint8_t* const pOutBuffer, size_t size) const NN_NOEXCEPT;
};

}}}; //nn::dns::parser
