﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/dns/parser/dns_ParserDependencies.h>
#include <nn/dns/parser/dns_ParserCommon.h>

/**
 * @file
 *
 * This file contains the definition of the @a Question class provided
 * by the Dns Parser library.
 */

/**
 * @namespace nn::dns::parser
 * @brief DNSパーサライブラリの名前空間。
 */
namespace nn { namespace dns { namespace parser {

/**
 * @brief The @a Question class represents a DNS question record as
 * described by The RFC.
 */
class Question
{
private:
    /**
     * @brief A pointer to the Message that contains this object.
     */
    const Message* m_pMessage;

    /**
     * @brief The block of memory that refers to the buffer
     * that created the @a Question with @ref Question::FromBuffer().
     */
    MemoryBlock m_Range;

    /**
     * @brief Indicates the object size or if it is uninitialized.
     */
    size_t m_DirtySize;

    /**
     * @brief The name of the DNS Question.
     */
    Label m_Name;

    /**
     * @brief Indicates the question type..
     */
    TypeConstant m_Type;

    /**
     * @brief Indicates the class of network.
     */
    ClassConstant m_Class;

public:

    /**
     * @brief Constructor.
     */
    Question();

    /**
     * @brief Copy constructor.
     *
     * @param rhs The object on the right hand side of the expression.
     */
    Question(const Question& rhs);

    /**
     * @brief Destructor.
     */
    ~Question();

    /**
     * @brief Assignment operator.
     *
     * @param rhs The object on the right hand side of the expression.
     */
    Question& operator=(const Question& rhs);

    /**
     * @brief Returns true if two @a Question objects are equal.
     *
     * @param[in] rhs The @a Question on the right hand side of
     * the expression.
     *
     * @return Returns true if both objects are content equivalent.
     */
    bool operator==(const Question& rhs) const NN_NOEXCEPT;

    /**
     * @brief Getter for m_pMessage.
     *
     * @return A reference to the message pointer.
     */
    const Message* & GetMessage() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_Name.
     *
     * @return A reference to the name label field.
     */
    Label& GetName() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_Type.
     *
     * @return A reference to the type field.
     */
    TypeConstant& GetType() NN_NOEXCEPT;

    /**
     * @brief Getter / setter for m_Class.
     *
     * @return A reference to the class field.
     */
    ClassConstant& GetClass() NN_NOEXCEPT;

    /**
     * @brief Initializes the @a Question.
     *
     * @details
     * Initializes a @a Question with the provided
     * nn::dns::parser::Message: zeroes out all fields, sets dirty
     * flag to zero, and point the @a Question at the
     * nn::dns::parser::Message.
     *
     * @param[in] pMessage The provided @a Message.
     */
    void Initialize(const Message* pMessage) NN_NOEXCEPT;

    /**
     * @brief Get the size of a @a Question for serialization.
     *
     * @return Returns error or success:
     *          - -1 : An error occurred.
     *          - >0 : The size of the object.
     */
    ssize_t SizeOf() const NN_NOEXCEPT;

    /**
     * @brief Reads a @a Question from the contents of a buffer in the
     * standard RFC format.
     *
     * @details It is the reverse of @ref Question::ToBuffer().
     *
     * @param[in] pBuffer The buffer that contains @a Question bytes.
     *
     * @param[in] size The size in bytes left in the buffer.
     *
     * @return Returns the number of bytes processed or -1 on error.
     *
     * Error cases include a null buffer pointer or parsing error.
     */
    ssize_t FromBuffer(const uint8_t* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief Write a @a Question to a buffer in The RFC standard
     * format.
     *
     * @details It is the reverse of @ref Question::FromBuffer().
     *
     * @param[out] pOutBuffer The buffer that receives @a Question bytes.
     *
     * @param[in] size The size of the buffer in bytes.
     *
     * @return Returns the number of bytes written to the buffer or -1
     * on error.
     *
     * Error cases include null buffer pointer, a buffer of
     * insufficient size, or a parsing error.
     */
    ssize_t ToBuffer(uint8_t* const pOutBuffer, size_t size) const NN_NOEXCEPT;

};

}}}; //nn::dns::parser
