﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/dhcps.h>

/**
 * @file
 *
 * @brief This is the public types used with the server.
 */

/**
 * @namespace nn::dhcps
 */
namespace nn { namespace dhcps {


/*
 * Constants
 */


/**
 * @brief This type is used to represent a hash over the bytes present
 * in the DHCP client identifier option bytes.
 */
typedef uint64_t ClientIdentifierHash;

/**
 * @brief It is RFC standard for a client to provide a client
 * identifier DHCP option. However not all clients do and the
 * server uses zero to indicate that a client identifier
 * DHCP option was not present and in this case using
 * @ref GetLeaseByDhcpClientIdentifierHash() to query for lease
 * information is not allowed, rather you must use
 * @ref GetLeaseByEthernetMacAddress() instead.
 */
const ClientIdentifierHash ClientIdentifierHashUnknown = 0;

/**
 * @brief The size in of an Ethernet MAC address in bytes.
 */
const unsigned int EthernetMacAddressSize = 6;

/**
 * @brief An Ethernet MAC address type.
 */
typedef uint8_t EthernetMacAddress[EthernetMacAddressSize];

/**
 * @brief The recommended T1 timer ratio.
 */
const double DefaultT1Ratio = 0.50;

/**
 * @brief The recommended T2 timer ratio.
 */
const double DefaultT2Ratio = 0.875;

/**
 * @brief This constant defines the maximum number of DNS servers that
 * the DHCP server sends to clients if so initialized.
 */
const unsigned int MaximumDnsServerCount = 2;


/*
 * Types
 */


/**
 * @brief This enum class contains all of the DHCP client states.
 */
enum class State : uint8_t
{
    /**
     * @brief The lease is not bound to a client and available for use.
     */
    Unbound   = 0,

    /**
     * @brief A lease was sent to the client in response to the DHCP
     * Offer message.
     */
    Offered   = 1,

    /**
     * @brief The lease is bound to a client and neither t1 or t2
     * timers are expired.
     */
    Bound     = 2,

    /**
     * @brief The lease is bound to a client  but the t1 timer expired.
     */
    Renewing  = 3,

    /**
     * @brief The lease is bound to a client but both t1 and t2 timers
     * expired.
     */
    Rebinding = 4,
};

/**
 * @brief The DhcpClientIdentity structure contains data about the
 * identity of a DHCP client.
 */
struct DhcpClientIdentity
{
    /**
     * @brief If a client identifier option is provided in the DHCP
     * message then this field is a hash over the bytes present in the
     * option. If no client identifier option is present then this
     * value is ClientIdentifierHashUnknown (0).
     */
    ClientIdentifierHash hash;

    /**
     * @brief This field is the client IP address from bootp or the
     * IP address assigned from the lease table.
     */
    nn::socket::InAddr ipAddress;

    /**
     * @brief This is the Ethernet MAC address bytes from the client
     * hardware address (chaddr) field in bootp.
     */
    EthernetMacAddress macAddress;
};

/**
 * @brief This structure represents a single lease record for the
 * lease table.
 */
struct DhcpLease
{
    /**
     * @brief This field contains information used to identify a client.
     */
    DhcpClientIdentity client;

    /**
     * @brief The DHCP client state value.
     */
    State state;

    /**
     * @brief the lease timer specified in seconds since system startup.
     *
     * @details
     * - When it elapses the lease state advances to the
     * @a State::Unbound state and can be reused.
     *
     * - It is also the number of seconds the lease extends when a
     * client renews or rebinds with a DHCP request message.
     */
    uint32_t lease;

    /**
     * @brief The "t1" timer specified in seconds since system startup.
     * When it elapses the lease state advances to the
     * @a State::Renewing state.
     */
    uint32_t t1;

    /**
     * @brief The "t2" timer specified in seconds since system startup.
     * When it elapses the lease state advances to the
     * @a State::Rebinding state.
     */
    uint32_t t2;
};

/**
 * @brief This enum class contains all of the events that the server
 * may send to the application.
 */
enum class Event : uint32_t
{
    /**
     * @brief The server was instructed to start.
     *
     * @details This event does not provide additional data.
     */
    OnStart = 1000,

    /**
     * @brief The server background thread is running.
     *
     * @details This event does not provide additional data.
     */
    OnRunning = 1001,

    /**
     * @brief The server was instructed to stop.
     *
     * @details This event does not provide additional data.
     */
    OnStopping = 1002,

    /**
     * @brief The DHCP thread stopped running.
     *
     * @details This event does not provide additional data.
     */
    OnStopped = 1003,

    /**
     * @brief The server detected a change in the network
     * configuration for the network interface provided to
     * @ref @Initialize(). This condition invalidates the network
     * and the it stops.
     *
     * @details This event does not provide additional data.
     */
    OnDetectedNetworkChange = 1004,

    /**
     * @brief The server encountered an recoverable error
     * and the it stops. Repeated recoverable errors might indicate
     * an unsupported client.
     *
     * @details This event does not provide additional data.
     */
    OnRecoverableError = 1005,

    /**
     * @brief The server encountered an unrecoverable
     * error and the it stops.
     *
     * @details This event does not provide additional data.
     */
    OnUnrecoverableError = 1006,

    /**
     * @brief The server offered a lease to a client in
     * response to a DHCP discover message.
     *
     * @details The data pointer for this event is a @a DhcpLease
     * structure.
     */
    OnLeaseOffered = 4000,

    /**
     * @brief The server received a DHCP discover message
     * for a lease that was already in the @a State::Offered
     * state and resent the lease.
     *
     * @details The data pointer for this event is a @a DhcpLease
     * structure.
     */
    OnLeaseOfferResent = 4001,

    /**
     * @brief The server received a DHCP discover message for a lease
     * in an advanced state. For example the lease is already
     * @a State::Bound, @a State::Renewing, or
     * @a State::Rebinding.
     *
     * @details The data pointer for this event is a @a
     * DhcpClientIdentity structure.
     */
    OnLeaseOfferAdvancedState = 4002,

    /**
     * @brief The server received a DHCP discover message for a
     * client that requested an available address. The address was
     * successfully reused.
     *
     * @details The data pointer for this event is a @a DhcpLease
     * structure.
     */
    OnLeaseOfferReused = 4003,

    /**
     * @brief The server offered a lease but the client did not
     * advance the lease to the @ref State::Bound state and so the
     * lease expired and was released back to the lease table.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseOfferExpired = 4004,

    /**
     * @brief A client declined a message from the server and there is
     * an lease associated with the same client and the server
     * expires the lease.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseDecline =  4005,

    /**
     * @brief The client declined a message but the server did not
     * find a valid lease in the lease table for the client.
     *
     * @details The data pointer for this event is a @ref
     * DhcpClientIdentity structure.
     */
    OnLeaseDeclineInvalid =  4006,

    /**
     * @brief The server received a DHCP inform message for a valid
     * lease.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseInform = 4007,

    /**
     * @brief The client sent a DHCP inform message to the server but
     * the server did not find a valid lease in the lease table.
     *
     * @details The data pointer for this event is a @ref
     * DhcpClientIdentity structure.
     */
    OnLeaseInformInvalid = 4008,

    /**
     * @brief The client sent a DHCP request message to the server to
     * acquire or renew a valid lease.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseRequest = 4009,

    /**
     * @brief The client sent a DHCP request to renew the lease but
     * the server did not find a valid lease in the lease table.
     *
     * @details The data pointer for this event is a @ref
     * DhcpClientIdentity structure.
     */
    OnLeaseRequestInvalid = 4010,

    /**
     * @brief The server advanced a lease state to the
     * @ref State::Renewing state because the t1 timer expired.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseRenewing = 4011,

    /**
     * @brief The server advanced a lease state to the
     * @ref State::Rebinding state because the t2 timer expired.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseRebinding = 4012,

    /**
     * @brief The server returned a lease to the lease table because
     * the lease expired.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseExpired = 4013,

    /**
     * @brief The client requested that the server release a lease
     * back to the lease table.
     *
     * @details The data pointer for this event is a @ref DhcpLease
     * structure.
     */
    OnLeaseRelease = 4014,

    /**
     * @brief The client requested that the server release a lease
     * back to the lease table but no valid lease was found.
     *
     * @details The data pointer for this event is a @ref
     * DhcpClientIdentity structure.
     */
    OnLeaseReleaseInvalid = 4015,

    /**
     * @brief The server allocated the last record in the lease
     * table.
     *
     * @details This event does not provide any additional data.
     */
    OnTableFull =  4016,

    /**
     * @brief The lease table was full but either the client released
     * or the server expired a lease and so now at least one lease is
     * available.
     *
     * @details This event does not provide any additional data.
     */
    OnTableAvailable =  4017,

    /**
     * @brief The server received a DHCP discover message but the table was full.
     *
     * @details The data pointer for this event is a @ref
     * DhcpClientIdentity structure.
     */
    OnDiscoverWhileTableFull =  4018,

    /**
     * @brief The client sent a message to the server that was not
     * handled.
     *
     * @details The data pointer for this event is a
     * @ref DhcpClientIdentity structure.
     */
    OnDhcpUnhandledMessage =  4019,
};

/**
 * @brief The type definition for the function pointer that is called
 * on server-to-application events.
 *
 * @details It is important not to tarry in the callback function
 * because the server is single threaded and the callback
 * essentially pauses server processing.
 *
 * @param pContext The @ref UserConfiguration::pContext value
 * configured by the application in @ref Initialize().
 *
 * @param event The type of event received.
 *
 * @param pData Points to a data that provides more information about
 * the event or nullptr if no data are provided for the event.
 */
typedef void (*ApplicationCallback)(void* pContext,
                                    Event event,
                                    const void* pData);

/**
 * @brief This structure is used to initialize the DHCP server.
 */
struct UserConfiguration
{
    /**
     * @brief The name of the interface to bind the server to
     * such as "wl0" or "usb0".
     */
    const char* pInterfaceName;

    /**
     * @brief The beginning of the DHCP range in network-byte order.
     */
    nn::socket::InAddr begin;

    /**
     * @brief The end of the DHCP range in network-byte order.
     */
    nn::socket::InAddr end;

    /**
     * @brief A function pointer that receives server-to-application
     * event.
     *
     * @details
     * The callback is required, it cannot be nullptr.
     */
    ApplicationCallback pCallback;

    /**
     * @brief The callback context pointer. Using nullptr is valid.
     * The server does not manage the pointer.
     */
    void* pContext;

    /**
     * @brief The amount of time in seconds that a DHCP network
     * configuration persists in the @a State::Offered state.
     *
     * @details
     * - This value cannot be zero.
     * - It is recommended that this value be small, on the order of
     * 1-10 seconds.
     */
    uint32_t offerSeconds;

    /**
     * @brief The length of the lease in seconds. It is also the
     * number of seconds the lease extends when a client renews the
     * lease with a DHCP request message.
     *
     * @details
     * - This value cannot be zero.
     * - It is recommended that this value be larger than
     * @a offerSeconds, approximately the time needed, or on the order
     * of 5-10 minutes (300 - 600s).
     */
    uint32_t leaseSeconds;

    /**
     * @brief A value between 0 and 1 that represents the DHCP t1
     * timer in terms of a ratio of the total lease time provided in
     * @a leaseSeconds.
     *
     * @details
     * - This value cannot be zero.
     * - This value must be less than @a t2Ratio.
     * - The recommended value is @a DefaultT1Ratio.
     */
    double t1Ratio;

    /**
     * @brief A value between 0 and 1 that represents the DHCP t2
     * timer in terms of a ratio of the total lease time provided in
     * @a leaseSeconds.
     *
     * @details
     * - This value cannot be zero.
     * - This value must be greater than @a t1Ratio.
     * - The recommended value is @a DefaultT2Ratio.
     */
    double t2Ratio;

    /**
     * @brief Indicates whether or not the @a gateway value should be
     * sent to clients.
     */
    bool hasGateway;

    /**
     * @brief The default gateway address in network-byte order that
     * is sent to clients if @a hasGateway is set to true.
     */
    nn::socket::InAddr gateway;

    /**
     * @brief The number of zero-filled DNS servers present in the
     * @a pDnsServers array.
     *
     * @details
     * The maximum value supported is @a MaximumDnsServerCount (2).
     */
    uint32_t dnsServerCount;

    /**
     * @brief A pointer to a zero-filled array that contains DNS
     * server IP addresses in network-byte order. A nullptr value is
     * allowed if @a dnsServerCount is zero. The maximum length
     * is @a MaximumDnsServerCount (2). These values are sent to
     * clients if @a dnsServerCount is greater than zero.
     */
    nn::socket::InAddr* pDnsServers;
};

}};
