﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <curl/curl.h>

#include <nn/nn_Allocator.h>
#include <nn/dauth/detail/dauth_Interface.sfdl.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/dauth/detail/dauth_ServiceImpl.h>
#include <nn/dauth/dauth_Result.h>
#include <nn/sf/sf_IServiceObject.h>
#include <nn/sf/sf_LmemUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace dauth {

const char ServiceName[] = "dauth:0";

struct ExecutionResource
{
    static const size_t BufferSize = 16 * 1024;
    NN_ALIGNAS(std::alignment_of<std::max_align_t>::value)
        char buffer[BufferSize];
    CURL* curlHandle;
};

struct ExecutorConfig
{
    static const int QueueCapacity = 32;
    using ResultOutOfQueueCapacity = detail::ResultOutOfQueueCapacity;
};
class Executor
    : public util::Executor<ExecutorConfig>
{
public:
    void Execute(ExecutionResource& resource) NN_NOEXCEPT
    {
        util::Executor<ExecutorConfig>::Execute(&resource, sizeof(resource));
    }
};

// サービス提供用のクラス
class ServiceResource
{
    // NOTE: 非同期処理を実装する場合、次を行う
    // - 非同期処理用のリソースを受け取ってタスクを実行する Executor を返す関数を足す

public:
    using IService = detail::IService;

private:
    template <size_t HeapSize>
    class ExpHeapMemoryResourceHolder
    {
        NN_DISALLOW_COPY(ExpHeapMemoryResourceHolder);
        NN_DISALLOW_MOVE(ExpHeapMemoryResourceHolder);

    private:
        typename std::aligned_storage<HeapSize>::type m_Storage;
        lmem::HeapHandle m_HeapHandle;
        util::optional<sf::ExpHeapMemoryResource> m_pResource;

    public:
        ExpHeapMemoryResourceHolder() NN_NOEXCEPT
        {
            m_HeapHandle = lmem::CreateExpHeap(&m_Storage, sizeof(m_Storage), 0);
            m_pResource.emplace(m_HeapHandle);
        }
        ~ExpHeapMemoryResourceHolder() NN_NOEXCEPT
        {
            m_pResource = util::nullopt;
            lmem::DestroyExpHeap(m_HeapHandle);
        }
        MemoryResource& GetMemoryResourceRef() NN_NOEXCEPT
        {
            return *m_pResource;
        }
    };

private:
    static const size_t AllocatorHeapSize = 256 * 1024u;

    ExpHeapMemoryResourceHolder<AllocatorHeapSize> m_MemoryResourceHolder;
    sf::EmplacedRef<IService, detail::ServiceImpl> m_pService;
    Executor m_Executor;

public:
    ServiceResource() NN_NOEXCEPT;
    sf::SharedPointer<IService> GetSharedService() NN_NOEXCEPT;
    Executor& GetExecutorRef() NN_NOEXCEPT;
};

}} // ~namespace nn::dauth

namespace nn { namespace dauth {

using DefaultServiceResource = ServiceResource;

inline ServiceResource::ServiceResource() NN_NOEXCEPT
{
    m_pService = sf::CreateSharedObjectEmplaced<IService, detail::ServiceImpl>(
        &m_MemoryResourceHolder.GetMemoryResourceRef(),
        m_MemoryResourceHolder.GetMemoryResourceRef(), m_Executor);
    NN_ABORT_UNLESS(m_pService);
}

inline sf::SharedPointer<ServiceResource::IService> ServiceResource::GetSharedService() NN_NOEXCEPT
{
    return m_pService;
}

inline Executor& ServiceResource::GetExecutorRef() NN_NOEXCEPT
{
    return m_Executor;
}

}} // ~namespace nn::dauth
