﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/dauth/dauth_Types.h>
#include <nn/util/util_Execution.h>

namespace nn { namespace dauth {

//! @name システム開発用の機能
//! @{

/**
   @brief       デバイス認証トークンを取得します。

   @param[out]  pOutExpiration  デバイス認証トークンの有効期限 (システム起動からの経過時間) が格納されます。
   @param[out]  pOutTokenLength デバイス認証トークンの長さ (終端含まず) が格納されます。
   @param[out]  buffer          デバイス認証トークンが格納されます。トークンはヌル文字で終端されます。
   @param[in]   bufferSize      buffer に書き込み可能なバイト数を指定します。
   @param[in]   clientId        デバイス認証トークンを要求するクライアントを識別するためのIDを指定します。
   @param[in]   ignoreCache     true を指定するとキャッシュを無視します。
   @param[in]   pCancelable     認証要求処理を中断するための nn::util::Cancelable オブジェクトへのポインタがあれば指定します。

   @return      処理結果
   @retval      nn::ResultSuccess                           正常に終了した。
   @retval      nn::dauth::ResultCancelled                  呼び出し元の要求に従って処理を中断した。
   @retval      nn::dauth::ResultNetworkCommunicationError  ネットワーク通信に関するエラー発生により失敗した。

   @pre
                - pOutExpiration != nullptr
                - pOutTokenLength != nullptr
                - buffer != nullptr
                - bufferSize >= @ref RequiredBufferSizeForDeviceAuthenticationToken

   @post
                - buffer に長さが *pOutTokenLength のデバイス認証トークンを保持する。

   @details     デバイス認証トークンを取得します。
                本関数で取得したデバイス認証トークンは有効時間に一定の猶予を持たせてキャッシュされます。
                このためデバイス認証トークンがキャッシュされている場合は、本関数は即時返ります。
                そうでない場合は、サーバーとの認証処理を行いキャッシュを作成したうえで返ります。

                ignoreCache に true を指定した場合、キャッシュを無視して必ずサーバーとの認証処理を行います。

   @see         nn::dauth::RequiredBufferSizeForDeviceAuthenticationToken
 */
Result AcquireDeviceAuthenticationToken(
    TimeSpan* pOutExpiration, int* pOutTokenLength, char* buffer, size_t bufferSize,
    uint64_t clientId, bool ignoreCache,
    util::Cancelable* pCancelable) NN_NOEXCEPT;

/**
   @brief       デバイス認証トークンの取得要求。

   @param[in]   clientId        デバイス認証トークンを要求するクライアントを識別するためのIDを指定します。
   @param[in]   refresh         true を指定するとキャッシュを無視します。
   @param[in]   pCancelable     認証要求処理を中断するための nn::util::Cancelable オブジェクトへのポインタがあれば指定します。

   @return      処理結果が返ります。
   @retval      nn::ResultSuccess                           正常に終了した。
   @retval      nn::dauth::ResultCancelled                  呼び出し元の要求に従って処理を中断した。
   @retval      nn::dauth::ResultNetworkCommunicationError  ネットワーク通信に関するエラー発生により失敗した。

   @details     デバイス認証トークンのキャッシュを作成します。
                本関数で作成したデバイス認証トークンのキャッシュは有効時間に一定の猶予を持たせて保持されます。
                このためデバイス認証トークンがキャッシュされている場合は、本関数は即時返ります。
                そうでない場合は、サーバーとの認証処理を行いキャッシュを作成したうえで返ります。
 */
Result EnsureDeviceAuthenticationTokenCache(uint64_t clientId, bool refresh, util::Cancelable* pCancelable) NN_NOEXCEPT;

/**
   @brief       CDN のエッジ認証に使用する認証トークンを取得します。
 */
Result AcquireEdgeToken(
    TimeSpan* pOutExpiration, int* pOutTokenLength, char* buffer, size_t bufferSize,
    uint64_t clientId,
    util::Cancelable* pCancelable) NN_NOEXCEPT;

/**
   @brief       CDN のエッジ認証に使用する認証トークンの取得要求。
 */
Result EnsureEdgeTokenCache(uint64_t clientId, util::Cancelable* pCancelable) NN_NOEXCEPT;

//! @}

}} // ~namespace nn::dauth
