﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Cpad ライブラリの API リスト。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include "cpad_Types.h"

namespace nn {
namespace cpad {

/**
 * @brief       Cpad ライブラリを初期化します。
 *
 * @post        Cpad ライブラリは初期化済の状態になります。
 *
 * @details     Cpad ライブラリを初期化します。
 *              本関数は任意のタイミングで繰り返し呼び出すことが可能です。
 *              本関数の呼び出し後に抜き差しが起きた場合であっても、本関数の呼出しは不要です。
 *              本関数はスレッドセーフです。
 *
 */
void InitializeCpad() NN_NOEXCEPT;

/**
 * @brief       Cpad の入力状態を取得します。
 *
 * @details     Cpad の入力状態を取得します
 *              入力状態の取得に失敗した場合、バッファ上の値は維持されます。
 *              クラシックコントローラが未初期化の場合、本関数呼出し時に暗黙的に
 *              クラシックコントローラの初期化処理が行われます。
 *              本関数呼び出し後に抜き差しが起きた場合であっても、本関数の呼出しは不要です。
 *
 * @param[out]  pOutState                       入力状態取得用のバッファ
 *
 * @return      実行結果を返します。
 * @retval      Result::IsSuccess               処理に成功しました。
 * @retval      ResultCpadNotFound              Cpad が接続されていません。
 *
 * @pre         - Cpad ライブラリは初期化済である必要があります。
 *              - pOutState != nullptr
 *
 * @post
 *              コントローラ入力の禁則処理
 *              - 十字ボタン左右の同時押し
 *              - 十字ボタン上下の同時押し
 *              - 座標 pOutState->stickL と pOutState->stickR は下記の条件を満たす正八角形の内側（境界含む）に位置
 *                  - 外接円の原点は (CpadStickOrigin, CpadStickOrigin)
 *                  - 外接円の半径は CpadStickMax - CpadStickOrigin
 *                  - 頂点の一つが (0, CpadStickMax - CpadStickOrigin) に位置
 */
Result GetCpadState(CpadState* pOutState) NN_NOEXCEPT;

/**
 * @brief       Cpad の入力状態列中に状態の変化が含まれるか否かを判定します。
 *
 * @details     指定の入力状態列上で、デジタルボタンの押下状態やスティックの傾きの変化が検出された場合 true を、それ以外の場合 false を返します。
 *              新しい入力状態ほど添字は小さくなります。
 *              入力状態間の時間間隔に制限はありません。
 *
 * @param[in]   pInStates                       入力状態の配列
 * @param[in]   statesCount                     入力状態の配列の長さ
 *
 * @return      入力状態列中に状態の変化が含まれるか否かを示す値を返します。
 * @retval      true                            入力状態列中で状態の変化が検出されました。
 * @retval      false                           入力状態列中で状態の変化が検出されませんでした。
 *
 * @pre         - Cpad ライブラリは初期化済である必要があります。
 *              - pInStates != nullptr
 *              - statesCount >= 0
 */
bool HasCpadChange(const CpadState *pInStates, int statesCount) NN_NOEXCEPT;

/**
 * @brief       Cpad の入力の有無を判定します。
 *
 * @details     入力状態上で、デジタルボタンの押下やスティックの傾きが検出された場合 true を、それ以外の場合 false を返します。
 *
 * @param[in]   state                           入力状態
 *
 * @return      入力の有無を示します。
 * @retval      true                            入力状態中に入力が検出されました。
 * @retval      false                           入力状態中に入力が検出されませんでした。
 *
 * @pre         Cpad ライブラリは初期化済である必要があります。
 */
bool HasCpadInput(CpadState state) NN_NOEXCEPT;

/**
 * @brief       スティックのキャリブレーションを行います。
 *
 * @details     本関数呼び出し時点での入力値を元に、スティックのキャリブレーションを行います。
 *              スティックがニュートラルな状態で本関数を呼び出すことでゼロ点の位置ずれが改善します。
 *              未キャリブレーション状態では、スティックに対して固定のゼロ点が設定されており、
 *              スティックごとの個体差によってゼロ点ずれが発生します。
 *              クラシックコントローラの抜き差しが行われた場合は、未キャリブレーション状態になります。
 *
 * @return      キャリブレーションに成功したかどうかを示します。
 * @retval      true                            キャリブレーションに成功しました。
 * @retval      false                           キャリブレーションに失敗しました。
 *
 * @pre         Cpad ライブラリは初期化済である必要があります。
 */
bool CalibrateCpad() NN_NOEXCEPT;

} // cpad
} // nn
