﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   USB Audio Class API
 */

#include <nn/usb/usb_Host.h>
#include <nn/usb/usb_Result.public.h>

namespace nn {
namespace cduac {

/**
 * @brief Size of descriptor buffer in InterfaceProfile, in bytes.
 */
const int DescriptorCountMax = 1024;


/**
 * @brief The InterfaceProfile used to store parameters for a given UAC audio interface.
 */
struct InterfaceProfile
{
    nn::usb::InterfaceQueryOutput   usbInterfaceParams;
    uint8_t                         audioControlInterfaceNumber;
    size_t                          descriptorsSize;
    uint8_t                         descriptors[DescriptorCountMax];
};


//--------------------------------------------------------------------------
//  Host class
//--------------------------------------------------------------------------
class Host
{

public:

    /**
     * @brief Initialize UAC host
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     *   @handleresult{ResultMemAllocFailure, Out of memory}
     * @endretresult
     *
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize()
     *   was called.
     *
     * @post
     *   Host is ready for operation.
     *
     * @details
     *   This method initializes internal data in Host object, and also USB host stack dependencies.
     */
    Result Initialize() NN_NOEXCEPT;

    /**
     * @brief Finalize UAC Host
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *
     * @post
     *   UAC host can no longer be used by this process.
     *
     * @details
     *   This method finalizes the Host object.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Create UAC interface availability event
     *
     * @param[out] pOutEvent
     *   Supplied system event is initialized.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Creation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Specified system event is now initialized.
     *
     * @details
     *   When an UAC interface becomes available, this system event will be signalled.
     */
    Result CreateInterfaceAvailableEvent(nn::os::SystemEventType* pOutEvent) NN_NOEXCEPT;

    /**
     * @brief Destroy UAC interface availability event
     *
     * @param[in] pInEvent
     *   Supplied system event is destroyed.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Destroy succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first. Specified system event should
     *   have been successfully created via prior call to
     *   CreateInterfaceAvailableEvent().
     *
     * @post
     *   Specified system event is now finalized.
     *
     * @details
     *   The system event is no longer receving signals, and should not be used.
     */
    Result DestroyInterfaceAvailableEvent(nn::os::SystemEventType* pInEvent) NN_NOEXCEPT;

    /**
     * @brief Query available UAC interfaces
     *
     * @param[out] pOutIfCount
     *   Pointer to returned interface count.
     *
     * @param[out] pOutBuffer
     *   Pointer to buffer, into which query output will be returned.
     *
     * @param[in] queryBufferSize
     *   Size of provided buffer referred to by pOutBuffer.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Internal state of USB service or device is not affected.
     *
     * @details
     *   Only UAC interfaces which are enumerated, available are returned.
     */
    Result QueryAvailableInterfaces(int32_t* pOutIfCount, nn::usb::InterfaceQueryOutput* pOutBuffer, size_t queryBufferSize) NN_NOEXCEPT;

    /**
     * @brief Get nn::usb::Host used by Host
     *
     * @return A pointer to the nn::usb::Host
     *
     * @details
     *   This function is used by UacParser::Initialize() and UacInterface::Initialize().
     */
    nn::usb::Host* GetUsbHost() NN_NOEXCEPT;

private:

    nn::usb::Host           m_UsbHost;
    nn::usb::DeviceFilter   m_DeviceFilter;

};


//--------------------------------------------------------------------------
//  Parser class
//--------------------------------------------------------------------------
class Parser
{

public:

    /**
     * @brief Create InterfaceProfile for specified USB device interface
     *
     * @param[out]  pOutInterfaceProfile
     *   Pointer to InterfaceProfile, into which the InterfaceProfile will be created.
     *
     * @param[in]  pHost
     *   Pointer to Host.
     *
     * @param[in]  pInInterfaceQueryOutput
     *   Pointer interface query information associated with the UAC device.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     *   @handleresult{ResultMemAllocFailure, Out of memory}
     *   @handleresult{ResultInterfaceInvalid}
     *   @handleresult{ResultInterfaceLimitExceeded}
     * @endretresult
     *
     * @pre
     *   pUsbHost is initialized.
     *
     * @details
     *   This function parses the device descriptors and creates the InterfaceProfile that represents a UAC device.
     */
    Result CreateInterfaceProfile(InterfaceProfile* pOutInterfaceProfile, Host* pHost, nn::usb::InterfaceQueryOutput* pInInterfaceQueryOutput) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioControlInputTerminal within the specified InterfaceProfile.
     *
     * @param[out]  pOutInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioControlInputTerminal descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioControlInputTerminal* GetInputTerminal(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioControlOutputTerminal within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioControlOutputTerminal descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioControlOutputTerminal* GetOutputTerminal(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioControlFeatureUnit within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioControlFeatureUnit descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioControlFeatureUnit* GetFeatureUnit(InterfaceProfile* pInterfaceProfile) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to nn::usb::UsbInterfaceDescriptor within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @param[in]  bAlternateSetting
     *   Alterate setting to get descriptor for.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to nn::usb::UsbInterfaceDescriptor descriptor if it is found in the InterfaceProfile or NULL.
     */
    nn::usb::UsbInterfaceDescriptor* GetInterface(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioStreamingGeneral within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @param[in]  bAlternateSetting
     *   Alterate setting to get descriptor for.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioStreamingGeneral descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioStreamingGeneral* GetAudioStreamingGeneral(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioStreamingFormatType within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @param[in]  bAlternateSetting
     *   Alterate setting to get descriptor for.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioStreamingFormatType descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioStreamingFormatType* GetAudioStreamingFormatType(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to nn::usb::UsbEndpointDescriptor within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @param[in]  bAlternateSetting
     *   Alterate setting to get descriptor for.
     *
     * @param[in]  endpointIndex
     *   Index of endpoint array.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to nn::usb::UsbEndpointDescriptor descriptor if it is found in the InterfaceProfile or NULL.
     */
    nn::usb::UsbEndpointDescriptor* GetEndpoint(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting, uint8_t endpointIndex) NN_NOEXCEPT;

    /**
     * @brief Retrieves pointer to AudioStreamingEndpointGeneral within the specified InterfaceProfile.
     *
     * @param[in]  pInterfaceProfile
     *   Pointer to InterfaceProfile associated with the UAC device.
     *
     * @param[in]  bAlternateSetting
     *   Alterate setting to get descriptor for.
     *
     * @param[in]  endpointIndex
     *   Index of endpoint array.
     *
     * @pre
     *   InterfaceProfile is created by calling CreateInterfaceProfile.
     *
     * @details
     *   This function returns pointer to AudioStreamingEndpointGeneral descriptor if it is found in the InterfaceProfile or NULL.
     */
    AudioStreamingEndpointGeneral* GetAudioStreamingEndpointGeneral(InterfaceProfile* pInterfaceProfile, uint8_t bAlternateSetting, uint8_t endpointIndex) NN_NOEXCEPT;

private:

    NN_ALIGNAS(4096) uint8_t        m_CtrlBuffer[4096];
    nn::usb::HostInterface          m_CtrlSession;

    uint8_t GetNodeId(AudioDescriptor *pAudioDescriptor) NN_NOEXCEPT;
    uint8_t GatherPins(uint8_t sourceId, uint8_t *p) NN_NOEXCEPT;
    AudioDescriptor * GetNode(uint8_t id, uint8_t *p) NN_NOEXCEPT;
    AudioDescriptor * GetReferencingNode(uint8_t sourceId, uint8_t *p) NN_NOEXCEPT;
    uint8_t GetSourceId(AudioDescriptor *pAudioDescriptor) NN_NOEXCEPT;
    void TraceInputToOutput(AudioDescriptor *pAudioDescriptor, InterfaceProfile *pOutInterfaceProfile) NN_NOEXCEPT;
    void TraceOutputToInput(AudioDescriptor *pAudioDescriptor, InterfaceProfile *pOutInterfaceProfile) NN_NOEXCEPT;

    uint8_t * GetNextDescriptor(void *p) NN_NOEXCEPT;
    uint8_t GetDescriptorLength(void *p) NN_NOEXCEPT;
    uint8_t GetDescriptorType(void *p) NN_NOEXCEPT;
    uint8_t GetAudioDescriptorSubType(void *p) NN_NOEXCEPT;

    uint8_t GetAudioControlInterfaceNumber(uint8_t *p) NN_NOEXCEPT;
    Result ParseDescriptors(InterfaceProfile *pOutInterfaceProfile, nn::usb::InterfaceQueryOutput* pInterface) NN_NOEXCEPT;
    void StoreDescriptor(InterfaceProfile *pInterfaceProfile, void *p) NN_NOEXCEPT;
    uint8_t GetInputTerminalIndex(uint8_t bInterfaceNumber, uint8_t *p) NN_NOEXCEPT;
    nn::usb::UsbInterfaceDescriptor* GetInterfaceDescriptor(uint8_t bInterfaceNumber, uint8_t bAletrnateSetting, uint8_t *p) NN_NOEXCEPT;
    bool IsUsbInterfaceDescriptor(uint8_t *p) NN_NOEXCEPT;
};


//--------------------------------------------------------------------------
//   Interface class
//--------------------------------------------------------------------------
class Interface
{

public:

    /**
     * @brief Initializes session for specified UAC device interface
     *
     * @param[in]  pHost
     *   Pointer to Host.
     *
     * @param[in]  pInInterfaceProfile
     *   Pointer to InterfaceProfile that was created by UacParser::CreateInterfaceProfile.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     *   @handleresult{ResultMemAllocFailure, Out of memory}
     *   @handleresult{ResultInterfaceInvalid}
     *   @handleresult{ResultInterfaceLimitExceeded}
     * @endretresult
     *
     * @pre
     *   Initialize() has not already been called, or if it was then Finalize()
     *   was called.
     *
     * @details
     *   By doing this operation, this host class driver now "owns" this
     *   UAC device interface.
     */
    Result Initialize(Host* pHost, InterfaceProfile* pInInterfaceProfile) NN_NOEXCEPT;

    /**
     * @brief Terminates and destroys previously established UAC device interface session.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Query succeeded}
     *   @handleresult{ResultInterfaceInvalidState}
     * @endretresult
     *
     * @pre
     *   Initialize() has already been called.
     *
     * @details
     *   Terminates previously established device interface session.
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * @brief Get the system event for Interface State Change Event
     *
     * @return A pointer to the system event
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   The returned pointer points to a system event object managed by this
     *   object. When this interface experiences a change in state, this system
     *   event will be signalled.
     *
     *   The returned system event should be cleared manually.
     *
     *   Note that the pointer returned will become invalid if this
     *   HostInterface object is finalized or destroied.
     */
    nn::os::SystemEventType* GetStateChangeEvent() NN_NOEXCEPT;

    /**
     * @brief Get the nn::usb::HostInterface associated with this UacInetrface
     *
     * @return A pointer to the nn::usb::HostInterface.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieves nn::usb::HostInterface associated with this UACInterface.
     */
    nn::usb::HostInterface* GetHostInterface() NN_NOEXCEPT;

    /**
     * @brief Get the InterfaceProfile associated with this UacInetrface
     *
     * @return A pointer to the InterfaceProfile.
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieves InterfaceProfile associated with this UACInterface.
     */
    InterfaceProfile* GetInterfaceProfile() NN_NOEXCEPT;

    /**
     * @brief Submit Set Audio Control request
     *
     * @param[out]  bRequest
     *   nn::uac::Request::Request_Set* as defined by UAC spec.
     *
     * @param[out]  wValue
     *   wValue for SetRequest as defined in UAC spec per entity.
     *
     * @param[in]  id
     *   Entity id.
     *
     * @param[in]  wLength
     *   Length in bytes of data to write.
     *
     * @param[in]  pData
     *   Pointer to data.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Data was written to device.
     *
     * @details
     *   Submit a Set Audio Control control request, where data is to be sent to the device.
     */
    Result SetAudioControl(uint8_t bRequest, uint16_t wValue, uint8_t id, uint16_t wLength, void *pData) NN_NOEXCEPT;

    /**
     * @brief Submit Get Audio Control request
     *
     * @param[out]  bRequest
     *   nn::uac::Request::Request_Get* as defined by UAC spec.
     *
     * @param[out]  wValue
     *   wValue for SetRequest as defined in UAC spec per entity.
     *
     * @param[in]  id
     *   Entity id.
     *
     * @param[in]  wLength
     *   Length in bytes of data to read.
     *
     * @param[in]  pData
     *   Pointer to data.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Data was read from device.
     *
     * @details
     *   Submit a Get Audio Control control request, where data is to be retrieved from the device.
     */
    Result GetAudioControl(uint8_t bRequest, uint16_t wValue, uint8_t id, uint16_t wLength, void *pData) NN_NOEXCEPT;

    /**
     * @brief Submit Set Endpoint Control request
     *
     * @param[out]  bRequest
     *   nn::uac::Request::Request_Set* as defined by UAC spec.
     *
     * @param[out]  cs
     *   nn::uac::Endpoint::Endpoint_* as defined by UAC spec.
     *
     * @param[in]  endpoint
     *   Endpoint address.
     *
     * @param[in]  wLength
     *   Length in bytes of data to write.
     *
     * @param[in]  pData
     *   Pointer to data.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Data was written to device.
     *
     * @details
     *   Submit a Set Endpoint Control control request, where data is to be sent to the device.
     */
    Result SetEndpointControl(uint8_t bRequest, uint8_t cs, uint8_t endpoint, uint16_t wLength, void *pData) NN_NOEXCEPT;

    /**
     * @brief Submit Get Endpoint Control request
     *
     * @param[out]  bRequest
     *   nn::uac::Request::Request_Get* as defined by UAC spec.
     *
     * @param[out]  cs
     *   nn::uac::Endpoint::Endpoint_* as defined by UAC spec.
     *
     * @param[in]  endpoint
     *   Endpoint address.
     *
     * @param[in]  wLength
     *   Length in bytes of data to read.
     *
     * @param[in]  pData
     *   Pointer to data.
     *
     * @retresult
     *   @handleresult{ResultSuccess, Operation succeeded}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @post
     *   Data was read from device.
     *
     * @details
     *   Submit a Get Endpoint Control control request, where data is to be retrieved from the device.
     */
    Result GetEndpointControl(uint8_t bRequest, uint8_t cs, uint8_t endpoint, uint16_t wLength, void *pData) NN_NOEXCEPT;

    /**
     * @brief Register sMMU and report buffers
     *
     * @param[in] smmuBuffer
     *   The data buffer you are going to read from / write to with PostTransferAsync()
     *
     * @param[in] smmuBufferSize
     *   Size of the sMMU buffer
     *
     * @param[in] reportBuffer
     *   The buffer used to hold transfer report
     *
     * @param[in] reportBufferSize
     *   Size of the report buffer
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   This call statically maps the smmuBUffer, which eliminates the need for
     *   map & unmap for each PostTransferAsync(). It also prepares a dedicated
     *   buffer to hold transfer reports, which eliminates the need of SF call
     *   for GetXferReport().
     *
     *   This is not necessary for the normal operation of the UAC device. Use
     *   it only if you want to minimize the CPU load.
     */
    Result RegisterBuffer(void *smmuBuffer,   uint32_t smmuBufferSize,
                          void *reportBuffer, uint32_t reportBufferSize) NN_NOEXCEPT;

    /**
     * @brief Performs standard set-interface command
     *
     * @param[in]  altSetting
     *   Alternate setting to be applied
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     *   @handleresult{ResultInvalidAlternateSetting}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Performs standard set-interface command.
     */
    Result SetAltSetting(uint8_t alternateSetting, nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor, uint32_t maxUrb, uint16_t maxXferSize) NN_NOEXCEPT;

    /**
     * @brief Retrieve the current USB host controller frame count
     *
     * @param[out]  pFrameNumber
     *   Returned frame number
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   Retrieve the current USB host controller frame count. This is usually
     *   relevant only for isochronous scheduling.
     */
    Result GetCurrentFrame(uint32_t *pFrameNumber) NN_NOEXCEPT;

    /**
     * @brief Post a batch of transfers on this endpoint
     *
     * @param[in] pBuffer
     *   Pointer to data buffer
     *
     * @param[in] pBytes
     *   Pointer to array of transfer size, one entry per Isoc frame (1ms)
     *
     * @param[in] usbFrames
     *   Number of entries in array
     *
     * @param[in] startOnFrame
     *   Start transfer on specified USB frame
     *
     * @param[in] context
     *   Context passed to nn::usb::XferReport.context upon completion
     *
     * @retresult
     *   @handleresult{nn::usb::ResultResourceBusy}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *   The buffer must be aligned to nn::usb::HwLimitDmaBufferAlignmentSize.
     *
     * @details
     *   Post an array of Isoc transfers on the USB bus. Each Isco frame is 1ms.
     */
    Result PostTransferAsync(void *pBuffer, uint32_t *pBytes, uint32_t usbFrames, uint32_t startOnFrame, uint32_t context) NN_NOEXCEPT;

    /**
     * @brief Gets the buffer completion event
     *
     * @retresult
     *   @handleresult{ResultSuccess}
     * @endretresult
     *
     * @pre
     *   Initialize() must be called once first.
     *
     * @details
     *   This function returns an event that is signalled when a posted buffer completes.
     *   The event needs to be cleared manually.
     */
    os::SystemEventType* GetCompletionEvent() NN_NOEXCEPT;

    /**
     * @brief Poll the status of submitted transfers
     *
     * @param[out] pOutCount
     *   Number of reports returned
     *
     * @param[out] pOutReport
     *   Pointer to the buffer where the reports will be stored
     *
     * @param[out] count
     *   Number of reports requested
     *
     * @retresult{Always success}
     * @endretresult
     *
     * @details
     *   Get the status report of completed transfers.
     */
    Result GetXferReport(uint32_t *pOutCount, nn::usb::XferReport *pOutReport, uint32_t count) NN_NOEXCEPT;


private:

    NN_ALIGNAS(4096) uint8_t    m_CtrlBuffer[4096];
    nn::os::MutexType           m_CtrlBufferMutex;
    InterfaceProfile            m_InterfaceProfile;
    nn::usb::HostInterface      m_UsbHostInterface;
    nn::usb::HostEndpoint       m_UsbHostEndpoint;
    uint8_t                     m_AlternateSetting;
};


} // end of namespace cduac
} // end of namespace nn
