﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Btm Types
 */

#pragma once

//[Todo]型を共用するか決める。当てはまるのは、Bluetoothアドレスくらい。
#include <nn/bluetooth/bluetooth_Types.h>
#include <nn/btm/user/btm_UserTypes.h>

#include <nn/settings/system/settings_BluetoothDevices.h>

namespace nn { namespace btm {

enum BtmState
{
    BtmState_NotInitialized,     //BTMは未初期化。基盤検査でのみこのステートに遷移し、停滞する
    BtmState_RadioOff,           //BTMはRadioOffモード
    BtmState_MinorSlept,         //BTMはマイナースリープステート。PageScan無効、LLRScan有効。API有効
    BtmState_RadioOffMinorSlept, //BTMはRadioOffモードかつマイナースリープステート
    BtmState_Slept,              //BTMはスリープステート
    BtmState_RadioOffSlept,      //BTMはRadioOffモードかつスリープステート
    BtmState_Initialized,        //BTMはリクエスト待ち状態。Bluetooth層が動作中（例：Discovery, Bond）でも、このステートに停滞する
    BtmState_Finalized,          //BTMは末期化。システムシャットダウン処理中のみこのステートに遷移し、停滞する
};

enum BluetoothMode
{
    BluetoothMode_Active,
    BluetoothMode_Auto,
};

enum WlanMode
{
    WlanMode_Local4,
    WlanMode_Local8,
    WlanMode_None,
    WlanMode_User8,
};

enum Profile
{
    Profile_None,
    Profile_Hid,
};

enum SniffMode
{
    SniffMode_5ms,
    SniffMode_10ms,
    SniffMode_15ms,
    SniffMode_Active,
};

//1デバイスに対するTx,Rxスロットの合算値
enum SlotMode
{
    SlotMode_2,
    SlotMode_4,
    SlotMode_6, //ユースケース無し
    SlotMode_Active,
};

// 1 BLEデバイス（Gatt Server）に対するスロットの合算値
enum CeLength : uint8_t
{
    CeLength_Invalid    = 0,    // BLE の接続がない場合
    CeLength_2          = 2,
    CeLength_4          = 4,
    CeLength_8          = 8,
};

static const int SIZE_OF_BDADDRESS = 6;
typedef nn::bluetooth::BluetoothAddress BdAddress;
/*struct BdAddress
{
    uint8_t address[SIZE_OF_BDADDRESS];
};*/

static const int SIZE_OF_BDNAME = 32;
struct BdName
{
    uint8_t name[SIZE_OF_BDNAME];//NULL終端
};

static const int SIZE_OF_COD = 3;
struct ClassOfDevice
{
    uint8_t cod[SIZE_OF_COD];
};

static const int SIZE_OF_LINKKEY = 16;
struct LinkKey
{
    uint8_t key[SIZE_OF_LINKKEY];
};

//ZeroRetransmission指定
static const int COUNT_OF_ZR_LIST = 16;
struct ZeroRetransmissionList{
    uint8_t enabledReportIdCount;
    uint8_t enabledReportId[COUNT_OF_ZR_LIST];
};


//スロットサイズ指定
struct DeviceSlotMode{
    BdAddress bdAddress;
    SlotMode slotMode;
};

static const int COUNT_OF_DSM_LIST = 8;
struct DeviceSlotModeList{
    uint8_t deviceCount;
    DeviceSlotMode device[COUNT_OF_DSM_LIST];
};

struct HostDeviceProperty
{
    BdAddress bdAddress;
    ClassOfDevice classOfDevice;
    BdName bdName;
    uint8_t featureSet;
};

//Discovery結果
struct DeviceProperty
{
    BdAddress bdAddress;
    ClassOfDevice classOfDevice;
    BdName bdName;
};

static const int COUNT_OF_DP_LIST = 15;
struct DevicePropertyList
{
    uint8_t deviceCount;
    DeviceProperty device[COUNT_OF_DP_LIST];
};


//Pairing結果
struct HidDeviceInfo
{
    uint16_t vid;
    uint16_t pid;
};

static const int SIZE_OF_DI_UNION = 32;
struct DeviceInfo
{
    BdAddress bdAddress;
    ClassOfDevice classOfDevice;
    BdName bdName;
    LinkKey linkKey;
    Profile profile;
    union
    {
        HidDeviceInfo hidDeviceInfo;
        uint8_t dummy[SIZE_OF_DI_UNION];
    };
    //[Todo]tBSA_SEC_ADD_DEV、tBSA_HH_ADD_DEV、BluetoothDevicesSettingsをもとに、必要なパラメータを後で足す
};

static const int COUNT_OF_DI_LIST = nn::settings::system::BluetoothDevicesSettingsCountMax;
struct DeviceInfoList
{
    uint8_t deviceCount;
    DeviceInfo device[COUNT_OF_DI_LIST];
};


//接続状況
struct HidDeviceCondition
{
    SniffMode sniffMode;
    SlotMode slotMode;
    bool isLargeSlotRequired;
    bool isBurstMode;
    ZeroRetransmissionList zeroRetransmissionList;
    uint16_t vid;
    uint16_t pid;
};

static const int SIZE_OF_DC_UNION = 64;
struct DeviceCondition
{
    BdAddress bdAddress;
    Profile profile;
    BdName bdName;
    union
    {
        HidDeviceCondition hidDeviceCondition;
        uint8_t dummy[SIZE_OF_DC_UNION];
    };
};

static const int COUNT_OF_DC_LIST = 8;
struct DeviceConditionList
{
    BluetoothMode bluetoothMode;
    WlanMode wlanMode;
    bool isSlotSavingForPairing;
    bool isSlotSaving;
    bool isLargeSlotRequiredForBle;
    uint8_t deviceCountCapacity;
    uint8_t deviceCount;
    DeviceCondition device[COUNT_OF_DC_LIST];
};

struct GattServer
{
    BdAddress   address;
    uint32_t    connectionHandle;
};

struct GattClient
{
    uint8_t     handle;
    GattServer  connectedServer;

    uint32_t    connectionInterval;
    uint16_t    slaveLatency;
    uint16_t    supervisionTimeout;
    uint16_t    mtu;
    CeLength    maxCeLength;    //!< 2, 4, or 8
};

struct GattClientConditionList
{
    GattClient  gattClients[nn::bluetooth::BLE_GATT_CLIENT_NUM_MAX];

    uint8_t     deviceCount;            //!< 現在の接続台数
    uint8_t     deviceCountCapacity;    //!< 許容する接続台数

    bool        isLargeCeLengthRequired;    //!< CeLength_8 を要求する接続が存在するか
};

//----------------------------------------
// For debug
//----------------------------------------
enum TsiMode
{
    TsiMode_0Fd3Td3Si10 = 0,
    TsiMode_1Fd1Td1Si5  = 1,
    TsiMode_2Fd1Td3Si10 = 2,
    TsiMode_3Fd1Td5Si15 = 3,
    TsiMode_4Fd3Td1Si10 = 4,
    TsiMode_5Fd3Td3Si15 = 5,
    TsiMode_6Fd5Td1Si15 = 6,
    TsiMode_7Fd1Td3Si15 = 7,
    TsiMode_8Fd3Td1Si15 = 8,
    TsiMode_9Fd1Td1Si10 = 9,
    TsiMode_10Fd1Td1Si15 = 10,
    TsiMode_Active = 0xFF,
};

static const int COUNT_OF_GI_LIST = 16;
struct GeneralInfoList{
    uint8_t count;
    uint8_t info[COUNT_OF_GI_LIST];
};

}}
