﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief 起動タイマーを操作する機能の API 宣言。
 */

#pragma once

#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>

#include "bpc_WakeupConfigTypes.h"

namespace nn { namespace bpc {

//! @name ウェイク設定
//! @{

/**
 * @brief ウェイク設定関連の機能ブロックを初期化します。
 *
 * @post
 *  対象の機能ブロックが初期化済の状態になります。
 *
 * @details
 *  ウェイク設定関連の機能ブロックを初期化します。 @n
 *  本関数の呼び出し回数はライブラリ内部でカウントされます。
 */
void InitializeWakeupConfig() NN_NOEXCEPT;

/**
 * @brief ウェイク設定関連の機能ブロックを終了します。
 *
 * @post
 *  対象の機能ブロックが初期化前の状態になります。
 *
 * @details
 *  ウェイク設定関連の機能ブロックを終了します。 @n
 *  初期化を行った回数分、本関数が呼ばれた場合にのみ、対象の機能ブロックが初期化前の状態になります。
 */
void FinalizeWakeupConfig() NN_NOEXCEPT;

/**
 * @brief 起動タイマーを有効にします。
 *
 * @param[out]  pOutTimerHandle     起動タイマーのハンドル
 * @param[in]   interval            起動までの時間
 * @param[in]   type                タイマーのタイプ
 *
 * @retresult
 *   @handleresult{nn::bpc::ResultMaxTimers}
 * @endretresult
 *
 * @pre
 *  - ウェイク設定関連機能ブロックが初期化済
 *
 * @details
 *  起動タイマーを有効にし、発動までの時間を設定します。
 *  本関数を呼び出した時刻から @a interval だけ経過した時刻になったとき、スリープ状態から起床します。@n
 *  起床予定時刻を過ぎてから初めてスリープした場合は、スリープ直後に起床します。 @n
 *  設定可能な時刻の範囲は、秒数にして [nn::bpc::MinWakeupIntervalInSeconds, nn::bpc::MaxRtcTimeInSeconds) です。 @n
 *  タイマーが複数設定された場合、設定された中で最も近い未来を指定したものが実際に有効になります。@n
 *  同じ時間でタイプの異なるタイマーが設定された場合は、 @ref nn::bpc::WakeupTimerType の各列挙子の優先度定義に基づいて選択されます。
 *  スリープ状態からの復帰後は、その復帰や起動の実際の要因に関わらず、すべての起動タイマーは破棄され、ハンドルは無効になります。
 */
nn::Result CreateWakeupTimer(int *pOutTimerHandle, nn::TimeSpan interval, WakeupTimerType type) NN_NOEXCEPT;

/**
 * @brief 廃止予定です。 CreateWakeupTimer(int*, nn::TimeSpan, WakeupTimerType) を使用してください。
 */
NN_DEPRECATED nn::Result CreateWakeupTimer(int *pOutTimerHandle, nn::TimeSpan interval) NN_NOEXCEPT;

/**
 * @brief 起動タイマーをキャンセルします。
 *
 * @param[in]   timerHandle         起動タイマーのハンドル
 *
 * @retresult
 *   @handleresult{nn::bpc::ResultNotRegistered}
 * @endretresult
 *
 * @pre
 *  - ウェイク設定関連機能ブロックが初期化済
 *
 * @details
 *  指定された起動タイマーをキャンセルします。 @n
 *  一度起動タイマーをセットした後、スリープまたはシャットダウンが発動する前にタイマーを再び無効にしたい場合に使用してください。 @n
 *  一度スリープまたはシャットダウンすると自動的にキャンセルされるため、その後この関数を呼ぶ必要はありません。 @n
 */
nn::Result CancelWakeupTimer(int timerHandle) NN_NOEXCEPT;

//! @}

}} // namepsace nn::bpc
