﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief       BLE に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/bluetooth/bluetooth_LeTypes.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultProprietary.h>

namespace nn { namespace bluetooth {

//! @name 独自に定義された BLE 機能に関する API
//! @{

/**
 * @brief       ペアリングされた BLE デバイスを対象とした BLE スキャンを開始し、ペアリングされた BLE デバイスの発見時に自動で接続する機能を有効化します。
 *
 * @details     開始された BLE スキャンは DisableBleAutoConnection() を呼ぶまで継続します。
 *              ペアリングされた BLE デバイスを発見すると自動的に接続処理が行われます。
 *              BLE デバイスと接続されると AcquireBleConnectionStateChangedEvent() で取得したイベントがシグナルされます。
 *              複数種類の BLE デバイスへの自動接続機能を有効にしたい場合、BLE デバイスに合わせたパラメータを使用して、本 API を複数回呼び出してください。
 *              他の BLE スキャンと同時に使用できます。
 *              異なるスキャンを並行して行っているかに関わらず、本 API によって同時にスキャン可能な BLE デバイスは 5 種類までです。
 *
 * @param[in]   scanParameter                   発見したいBLE デバイスのBLE アドバタイズパケットのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBleScanFull,             スキャン対象数の上限に達しています。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result EnableBleAutoConnection(const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされた BLE デバイスを対象とした BLE スキャンを停止し、ペアリングされた BLE デバイス発見時に自動で接続する機能を無効化します。
 *
 * @details     EnableBleAutoConnection() を複数回呼び出して、複数種類の BLE デバイスを対象とした自動接続を有効にしている場合、それらの設定が全て無効になります。
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultSystemBusy,              システムが他の処理中です。再試行により実行可能ですが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultGeneralError,            処理に失敗しました。再試行により操作に成功することがありますが、高頻度に再試行しないでください。}
 *  @handleresult{nn::bluetooth::ResultBluetoothOff,            Bluetooth が OFF になっているため、操作を実行できません。}
 * @endretresult
 */
nn::Result DisableBleAutoConnection() NN_NOEXCEPT;

/**
 * @brief       BLE ペアリングが完了した際にシグナルされるイベントを取得します。
 *
 * @details     以下の API の呼び出し後、実際のペアリング処理、ペアリング削除処理が完了する、もしくはエラーが発生したタイミングでイベントがシグナルされます。
 *              - PairGattServer()
 *              - UnpairGattServer()
 *
 * @param[out]  pOutSystemEvent                 取得されるシステムイベントへのポインタ
 *
 * @pre
 *              - pOutSystemEvent != nullptr
 *              - pOutSystemEvent は未初期化
 *
 * @post
 *              - pOutSystemEvent は初期化済み
 */
void AcquireBlePairingEvent(nn::os::SystemEventType* pOutSystemEvent) NN_NOEXCEPT;

/**
 * @brief       接続されている GATT Server と BLE ペアリングします。
 *
 * @details     本 API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              最大 @ref BlePairingCountMax 個の GATT Server と BLE ペアリングすることができます。
 *              BLE ペアリング数が最大に達している場合、最も古い BLE ペアリング情報を削除します。
 *              すでに BLE ペアリング情報が存在する場合、本体内の BLE ペアリング情報を更新します。
 *              ペアリング処理の成否を取得することはできません。
 *              GetPairedGattServerAddress() を使用して、ペアリング済みデバイスの一覧を取得してください。
 *              同時に処理可能なペアリング処理は 1 つまでです。
 *
 * @param[in]   connectionHandle                BLE ペアリングする GATT Server との接続ハンドル
 * @param[in]   scanParameter                   BLE ペアリングする GATT Server の BLE アドバタイズパケットのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultBlePairingNotSupported,  BLE ペアリング機能がサポートされていません。}
 *  @handleresult{nn::bluetooth::ResultBlePairingOnGoing,       BLE ペアリング処理中です。}
 * @endretresult
 */
nn::Result PairGattServer(uint32_t connectionHandle, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

/**
 * @brief       接続されている GATT Server との BLE ペアリング情報を削除します。
 *
 * @details     本 API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              本体内に BLE ペアリング情報が存在しないでも、GATT Server から BLE ペアリング情報を削除する処理を行います。
 *              BLE ペアリング削除処理の成否を取得することはできません。
 *              GetPairedGattServerAddress() を使用して、BLE ペアリング済みデバイスの一覧を取得してください。
 *              同時に処理可能な BLE ペアリング削除処理は 1 つまでです。
 *
 * @param[in]   connectionHandle                BLE ペアリングを削除する GATT Server との接続ハンドル
 * @param[in]   scanParameter                   BLE ペアリングを削除するする GATT Server の BLE アドバタイズパケットのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBleNotConnected,         接続が存在しません。}
 *  @handleresult{nn::bluetooth::ResultBlePairingNotSupported,  BLE ペアリング機能がサポートされていません。}
 *  @handleresult{nn::bluetooth::ResultBlePairingOnGoing,       BLE ペアリング処理中です。}
 * @endretresult
 */
nn::Result UnpairGattServer(uint32_t connectionHandle, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

/**
 * @brief       GATT Server との BLE ペアリング情報を削除します。
 *
 * @details     本 API で実施されるペアリングは、BLE の規格に準拠したペアリングではありません。
 *              詳細はドキュメントを参照してください。
 *              指定した Bluetooth アドレスを持つ GATT Server と接続している場合、本体、及び、GATT Server から BLE ペアリング情報を削除する処理を行います。
 *              指定した Bluetooth アドレスを持つ GATT Server と接続していない場合は、本体からのみ BLE ペアリング情報を削除します。
 *              BLE ペアリング情報が存在しない場合でも、指定したBluetooth アドレスを持つGATT Server と接続されいる場合、そのGATT Server からペアリング情報を削除する処理を行います。
 *              BLE ペアリング削除処理の成否を取得することはできません。
 *              GetPairedGattServerAddress() を使用して、BLE ペアリング済みデバイスの一覧を取得してください。
 *              同時に処理可能な BLE ペアリング削除処理は 1 つまでです。
 *
 * @param[in]   bluetoothAddress                BLE ペアリングを削除する GATT Server の Bluetooth アドレス
 * @param[in]   scanParameter                   BLE ペアリングを削除するする GATT Server のBLE アドバタイズパケットのパラメータ
 *
 * @retresult
 *  @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *  @handleresult{nn::bluetooth::ResultBlePairingNotSupported,  BLE ペアリング機能がサポートされていません。}
 *  @handleresult{nn::bluetooth::ResultBlePairingOnGoing,       BLE ペアリング処理中です。}
 * @endretresult
 */
nn::Result UnpairGattServer(const Address& bluetoothAddress, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

/**
 * @brief       ペアリングされている GATT Server のBluetooth アドレス一覧を取得します。
 *
 * @details     最大で @ref BlePairingCountMax 個の Bluetooth アドレスを取得します。
 *
 * @param[out]  pOutAddresses                   ペアリングされた GATT Server の Bluetooth アドレスを格納するための配列へのポインタ
 * @param[in]   count                           pOutAddresses の数
 * @param[in]   scanParameter                   ペアリング情報一覧を取得するする GATT Server の BLE アドバタイズパケットのパラメータ
 *
 * @pre
 *              - pOutAddresses != nullptr
 *
 * @return      取得したBluetooth アドレスの数
 *
 */
int GetPairedGattServerAddress(Address* pOutAddresses, int count, const BleAdvertisePacketParameter& scanParameter) NN_NOEXCEPT;

//! @}

}}  // namespace nn::bluetooth
