﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       GATT Descriptor クラスの宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/bluetooth/bluetooth_GattAttribute.h>

namespace nn { namespace bluetooth {

    class GattService;

    class GattCharacteristic;

    /**
     * @brief       GATT Descriptor を表すクラスです
     *
     * @details     GATT Attribute クラスを継承します
     */
    class GattDescriptor : public GattAttribute
    {
    public:
        /**
         * @brief           コンストラクタです
         *
         * @details         全パラメータが無効な値で初期化された GATT Descriptor オブジェクトを生成します
         */
        GattDescriptor() NN_NOEXCEPT;

        /**
         * @brief           コンストラクタです
         *
         * @details         与えられたパラメータで初期化された GATT Descriptor オブジェクトを生成します
         *
         * @param[in]       uuid                GATT Attribute UUID
         * @param[in]       handle              GATT Attribute Handle
         * @param[in]       connectionHandle    GATT Server との接続ハンドル
         */
        GattDescriptor(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle) NN_NOEXCEPT;

        /**
         * @brief       デストラクタです
         */
        virtual ~GattDescriptor() NN_NOEXCEPT;

        /**
         * @brief           自身が所属する GATT Service を取得します
         *
         * @param[out]      pOutService         GATT Service オブジェクトへのポインタ
         *
         * @pre
         *                  - pOutService != nullptr
         */
        void GetService(GattService* pOutService) const NN_NOEXCEPT;

        /**
         * @brief           自身が所属する GATT Characteristic を取得します
         *
         * @param[out]      pOutCharacteristic  GATT Characteristic オブジェクトへのポインタ
         *
         * @pre
         *                  - pOutCharacteristic != nullptr
         */
        void GetCharacteristic(GattCharacteristic* pOutCharacteristic) const NN_NOEXCEPT;

        /**
         * @brief           接続相手に送信する値を設定します
         *
         * @details         以下の API によって送信される値を設定します。
         *                  - WriteGattDescriptor()
         *
         *                  設定された値は、内部でコピーされます。
         *                  すでに値が設定されている場合、上書きします。
         *
         * @param[in]       pValue              設定する値
         * @param[in]       size                設定する値のサイズ
         *
         * @pre
         *                  - pValue != nullptr
         *                  - size <= @ref GattAttributeValueSizeMax
         */
        void SetValue(const void* pValue, size_t size) NN_NOEXCEPT;

        /**
         * @brief           接続相手に送信する値を取得します
         *
         * @details         以下の API によって送信される値を取得します。
         *                  - WriteGattDescriptor()
         *
         * @param[out]      pOutValue           取得する値を格納するための配列へのポインタ
         * @param[in]       size                pOutValue のサイズ
         *
         * @pre
         *                  - pOutValue != nullptr
         *
         * @return          取得した値のサイズ
         */
        size_t GetValue(void* pOutValue, size_t size) const NN_NOEXCEPT;

    private:
        size_t m_ValueLength;                          //!< m_Value に設定されている値のサイズです

        uint8_t m_Value[GattAttributeValueSizeMax];    //!< 接続相手に送信する値です
    };
}}  // namespace nn::bluetooth
