﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/bgsu/bgsu_Log.h>

namespace nn { namespace bgsu {

template<class BaseT, size_t StackSize = 64 * 1024>
class ThreadTemplate
{
public:

    explicit ThreadTemplate(const char* pName, int32_t priority = os::InvalidThreadPriority);
    virtual ~ThreadTemplate()
    {
    }

    Result Initialize();
    virtual void Finalize();

    virtual void RequestExit()
    {
        m_bExit = true;
    }

    bool IsAlive() const
    {
        return os::ThreadType::State_Started == m_thread._state;
    }

    bool IsExiting() const
    {
        return m_bExit;
    }

    os::ThreadId GetThreadId() const
    {
        NN_SDK_ASSERT(IsAlive());
        return os::GetThreadId(&m_thread);
    }

protected:
    virtual void ThreadBody() = 0;
    typedef ThreadTemplate<BaseT, StackSize> Base;

private:

    static void ThreadStarter(void* pThis);

    NN_OS_ALIGNAS_THREAD_STACK char m_stack[StackSize];

    os::ThreadType      m_thread;
    const char*         m_pName;
    volatile bool       m_bExit;
    NN_PADDING3;
    int32_t             m_priority;
};

template<class BaseT, size_t StackSize>
ThreadTemplate<BaseT, StackSize>::ThreadTemplate(const char* pName, int32_t priority)
    : m_pName(pName), m_priority(priority)
{
    std::memset(&m_thread, 0, sizeof(m_thread));
}


template<class BaseT, size_t StackSize>
Result ThreadTemplate<BaseT, StackSize>::Initialize()
{
    m_bExit = false;
    Result result = os::CreateThread(&m_thread, ThreadStarter, this, static_cast<void*>(m_stack), StackSize, m_priority);
    if (result.IsFailure())
    {
        NN_BGSU_ERROR("Failed to create thread for %s.(0x%08x)", m_pName, result.GetInnerValueForDebug());
        return result;
    }
    os::SetThreadNamePointer(&m_thread, m_pName);
    os::StartThread(&m_thread);
    return ResultSuccess();
}

template<class BaseT, size_t StackSize>
void ThreadTemplate<BaseT, StackSize>::Finalize()
{
    RequestExit();
    os::WaitThread(&m_thread);
    os::DestroyThread(&m_thread);
}

template<class BaseT, size_t StackSize>
void ThreadTemplate<BaseT, StackSize>::ThreadStarter(void* pThis)
{
    static_cast<Base*>(pThis)->ThreadBody();
}

}}
