﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/crypto.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace util {

/*!
    @brief      AES-CTR を利用した復号器のテンプレートクラスです。
*/
template <size_t KeySize>
class AesCtrDecryptor
{
public:
    typedef nn::crypto::AesEncryptor<KeySize> AesEncryptor;

    static const size_t BlockSize = nn::crypto::CtrDecryptor<AesEncryptor>::BlockSize;
    static const size_t IvSize = nn::crypto::CtrDecryptor<AesEncryptor>::BlockSize;

public:
    /*!
        @brief      コンストラクタです。
    */
    AesCtrDecryptor() NN_NOEXCEPT {};

public:
    /*!
        @brief      初期化します。

        @param[in]  key     暗号鍵。
        @param[in]  keySize 暗号鍵のサイズ。
        @param[in]  iv      初期ベクタ。
        @param[in]  ivSize  初期ベクタのサイズ。
    */
    void Initialize(const void* key, size_t keySize, const void* iv, size_t ivSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(keySize == KeySize);
        NN_SDK_REQUIRES(ivSize == IvSize);

        m_Aes.Initialize(key, keySize);
        m_CtrDecryptor.Initialize(&m_Aes, iv, ivSize);
    }

    /*!
        @brief      復号処理を行います。

        @param[in]  dest        出力データ。
        @param[in]  destSize    出力データのサイズ。
        @param[in]  src         入力データ。
        @param[in]  srcSize     入力データのサイズ。

        @return     復号したバイト数。
    */
    size_t Update(void* dest, size_t destSize, const void* src, size_t srcSize) NN_NOEXCEPT
    {
        return m_CtrDecryptor.Update(dest, destSize, src, srcSize);
    }

private:
    //
    AesEncryptor m_Aes;
    nn::crypto::CtrDecryptor<AesEncryptor> m_CtrDecryptor;
};

/*!
    @brief      AES-CTR (鍵長 128 ビット) の復号器です。
*/
typedef AesCtrDecryptor<16> AesCtrDecryptor128;

/*!
    @brief      AES-CTR (鍵長 192 ビット) の復号器です。
*/
typedef AesCtrDecryptor<24> AesCtrDecryptor192;

/*!
    @brief      AES-CTR (鍵長 256 ビット) の復号器です。
*/
typedef AesCtrDecryptor<32> AesCtrDecryptor256;

}}}}}
