﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      セッション管理モジュールです。
*/
class SessionManager
{
public:
    /*!
        @brief      管理するセッション数です。

        @details
                    クライアント向け IPC セッション＋BG 処理用です。
    */
    static const int SessionCountMax = detail::ipc::BcatServiceCountMax + 1;

    /*!
        @brief      セッションあたりの管理するタスク数です。
    */
    static const int TaskCountMaxPerSession = detail::ipc::DeliveryCacheProgressServiceCountPerProcess;

private:
    /*!
        @brief      コンストラクタです。
    */
    SessionManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static SessionManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(SessionManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      セッションを登録します。

        @param[out] outSessionId    セッション ID。

        @return     処理結果。

        @pre
            - outSessionId != nullptr
    */
    void Register(SessionId* outSessionId) NN_NOEXCEPT;

    /*!
        @brief      セッションの登録を解除します。

        @param[in]  sessionId   セッション ID。

        @return     処理結果。

        @pre
            - sessionId != InvalidSessionId
    */
    void Unregister(SessionId sessionId) NN_NOEXCEPT;

    /*!
        @brief      フォアグラウンド実行用タスクを追加します。

        @param[out] outTaskId   タスク ID。
        @param[in]  sessionId   セッション ID。
        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。
        @param[in]  dirName     ディレクトリ名。

        @return     処理結果。

        @pre
            - outTaskId != nullptr
            - sessionId != InvalidSessionId
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    ディレクトリ名に nullptr を指定した場合、ディレクトリ指定ではなく全体の同期が行われます。
    */
    nn::Result AddForegroundTask(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
        const char* dirName = nullptr) NN_NOEXCEPT;

    /*!
        @brief      バックグラウンド実行用タスクを追加します。

        @param[out] outTaskId   タスク ID。
        @param[in]  sessionId   セッション ID。
        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。
        @param[in]  dirName     ディレクトリ名。

        @return     処理結果。

        @pre
            - outTaskId != nullptr
            - sessionId != InvalidSessionId
            - appId != nn::ApplicationId::GetInvalidId()

        @details
                    ディレクトリ名に nullptr を指定した場合、ディレクトリ指定ではなく全体の同期が行われます。
    */
    nn::Result AddBackgroundTask(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
        const char* dirName = nullptr) NN_NOEXCEPT;

    /*!
        @brief      タスクを削除します。

        @param[in]  sessionId   セッション ID。
        @param[in]  taskId      タスク ID。

        @pre
            - sessionId != InvalidSessionId
            - taskId != InvalidTaskId
    */
    void RemoveTask(SessionId sessionId, TaskId taskId) NN_NOEXCEPT;

    /*!
        @brief      指定したセッションのタスクをすべてキャンセルします。

        @param[in]  sessionId   セッション ID。

        @pre
            - sessionId != InvalidSessionId
    */
    void CancelTask(SessionId sessionId) NN_NOEXCEPT;

private:
    //
    struct Task
    {
        TaskId taskId;
    };

    //
    struct Session
    {
        SessionId id;
        Task tasks[TaskCountMaxPerSession];
    };

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    Session m_Sessions[SessionCountMax];

private:
    //
    int Search(SessionId sessionId) const NN_NOEXCEPT;
    int SearchTask(int sessionIndex, TaskId taskId) const NN_NOEXCEPT;
    int SearchEmptyTask(int sessionIndex) const NN_NOEXCEPT;
    //
    nn::Result AddTask(TaskId* outTaskId, SessionId sessionId,
        nn::ApplicationId appId, uint32_t appVersion, const char* dirName, bool isForeground) NN_NOEXCEPT;
};

}}}}}
