﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      パスフレーズ管理モジュールです。
*/
class PassphraseManager
{
public:
    /*!
        @brief      一時的に保存するパスフレーズの数です。
    */
    static const int TemporaryCountMax = DeliveryCacheStorageManager::MountCountMax + 1;

private:
    /*!
        @brief      コンストラクタです。
    */
    PassphraseManager() NN_NOEXCEPT;

public:
    /*!
        @brief      インスタンスを取得します。

        @return     インスタンス。
    */
    static PassphraseManager& GetInstance() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(PassphraseManager, s_Instance);
        return s_Instance;
    }

public:
    /*!
        @brief      パスフレーズを取得します。

        @param[in]  passphrase  パスフレーズのバッファ。
        @param[in]  size        パスフレーズのバッファサイズ。
        @param[in]  appId       アプリケーション ID。

        @return     処理結果。

        @pre
            - passphrase != nullptr
            - size >= PassphraseLengthMax + 1
            - appId != nn::ApplicationId::GetInvalidId()
    */
    nn::Result Get(char* passphrase, size_t size, nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      パスフレーズを保存します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  passphrase  パスフレーズ。
        @param[in]  needsCommit 関数内でコミットするかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - passphrase != nullptr
            - データ配信キャッシュストレージをマウントしている。

        @details
                    アプリケーションがストレージにアクセスしている間、または、ダウンロード処理が行われている間、以下の理由により外部から本関数を呼び出してはいけません。

                    - アプリケーションがファイルを開いている間にコミットしてはいけない。
                    - ダウンロード処理中に意図しないタイミングでジャーナルを消費したりコミットしたりしてはいけない。

                    通常は、 @ref DeliveryCacheStorageManager::RequestSavePassphrase を使用してください。
    */
    nn::Result Save(nn::ApplicationId appId, const char* passphrase, bool needsCommit = true) NN_NOEXCEPT;

    /*!
        @brief      パスフレーズを削除します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  needsCommit 関数内でコミットするかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - データ配信キャッシュストレージをマウントしている。
    */
    nn::Result Delete(nn::ApplicationId appId, bool needsCommit = true) NN_NOEXCEPT;

    /*!
        @brief      テンポラリ領域にパスフレーズを追加します。

        @param[in]  appId       アプリケーション ID。
        @param[in]  passphrase  パスフレーズ。

        @return     追加したかどうか。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - passphrase != nullptr
    */
    bool AddTemporary(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT;

    /*!
        @brief      テンポラリ領域に追加したパスフレーズを削除します。

        @param[in]  appId   アプリケーション ID。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
    */
    void RemoveTemporary(nn::ApplicationId appId) NN_NOEXCEPT;

    /*!
        @brief      テンポラリ領域に存在するパスフレーズを保存します。

        @param[in]  appId           アプリケーション ID。
        @param[in]  needsCommit     関数内でコミットするかどうか。
        @param[in]  removeIfSuccess 成功時、テンポラリ領域から削除するかどうか。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - データ配信キャッシュストレージをマウントしている。
    */
    nn::Result SaveTemporary(nn::ApplicationId appId, bool needsCommit = true, bool removeIfSuccess = true) NN_NOEXCEPT;

private:
    //
    struct Record
    {
        nn::ApplicationId appId;
        char passphrase[PassphraseLengthMax + 1];
    };

private:
    //
    nn::os::Mutex m_Mutex;
    nn::os::Mutex m_MutexForTemporary;
    //
    Record m_Records[TemporaryCountMax];
    int m_Count;

private:
    //
    int SearchRecord(nn::ApplicationId appId) NN_NOEXCEPT;
};

}}}}}
