﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>
#include <nn/bcat/detail/service/core/bcat_DiffView.h>
#include <nn/bcat/service/bcat_ArchiveDownloader.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      ディレクトリのダウンロードを行うモジュールです。
*/
class DirectoryDownloader
{
public:
    /*!
        @brief      前回のダウンロード進捗取得コールバックです。
    */
    typedef void (*LastProgressCallback)(int64_t downloaded, const DirectoryName& dirName, void* param);

    /*!
        @brief      ファイルのダウンロード開始コールバックです。
    */
    typedef void (*StartDownloadFileCallback)(const DirectoryName& dirName, const FileName& fileName, int64_t fileSize, void* param);

    /*!
        @brief      ファイルのダウンロードコールバックです。
    */
    typedef void (*DownloadFileCallback)(int64_t downloaded, void* param);

public:
    /*!
        @brief      コンストラクタです。
    */
    DirectoryDownloader() NN_NOEXCEPT;

    /*!
        @brief      キャンセルイベントを設定します。

        @param[in]  pCancelEvent    キャンセルイベント。

        @pre
            - pCancelEvent != nullptr
    */
    void SetCancelEvent(nn::os::Event* pCancelEvent) NN_NOEXCEPT;

    /*!
        @brief      前回のダウンロード進捗取得コールバックを設定します。

        @param[in]  callback    コールバック。
        @param[in]  param       コールバックパラメーター。

        @pre
            - callback != nullptr
    */
    void SetLastProgressCallback(LastProgressCallback callback, void* param) NN_NOEXCEPT;

    /*!
        @brief      ファイルのダウンロード開始コールバックを設定します。

        @param[in]  callback    コールバック。
        @param[in]  param       コールバックパラメーター。

        @pre
            - callback != nullptr
    */
    void SetStartDownloadFileCallback(StartDownloadFileCallback callback, void* param) NN_NOEXCEPT;

    /*!
        @brief      ファイルのダウンロードコールバックを設定します。

        @param[in]  callback    コールバック。
        @param[in]  param       コールバックパラメーター。

        @pre
            - callback != nullptr
    */
    void SetDownloadFileCallback(DownloadFileCallback callback, void* param) NN_NOEXCEPT;

    /*!
        @brief      ダウンロード処理を行います。

        @param[in]  appId       アプリケーション ID。
        @param[in]  dirName     ディレクトリ名。
        @param[in]  passphrase  パスフレーズ。
        @param[in]  pConnection ネットワーク接続要求オブジェクト。
        @param[in]  buffer      ワークバッファ。
        @param[in]  bufferSize  ワークバッファのサイズ。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - passphrase != nullptr
            - pConnection != nullptr
            - buffer != nullptr
            - bufferSize > 0

        @details
                    本関数は、ダウンロードディレクトリのクリーンアップを行いません。@n
                    対象のディレクトリの更新を検知した場合、本関数を呼び出すよりも前にクリーンアップを行う必要があります。
    */
    nn::Result Download(nn::ApplicationId appId, const DirectoryName& dirName, const char* passphrase,
        nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT;

private:
    //
    nn::os::Mutex m_Mutex;
    //
    nn::ApplicationId m_AppId;
    char m_Passphrase[PassphraseLengthMax + 1];
    //
    nn::os::Event* m_pCancelEvent;
    //
    LastProgressCallback m_LastProgressCallback;
    void* m_LastProgressCallbackParam;
    //
    StartDownloadFileCallback m_StartDownloadFileCallback;
    void* m_StartDownloadFileCallbackParam;
    //
    DownloadFileCallback m_DownloadFileCallback;
    void* m_DownloadFileCallbackParam;
    //
    nn::bcat::service::ArchiveDownloader m_ArchiveDownloader;
    //
    DirectoryName m_DirName;
    ListFile m_Files[DeliveryCacheFileCountMaxPerDirectory];
    int m_FileCount;
    //
    int m_IncompleteFileIndex;
    //
    FileMetaView m_FileMetaView;
    FileDiffView m_FileDiffView;
    //
    FileMetaView m_DownloadFileMetaView;

private:
    //
    nn::Result StepSetup(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    nn::Result StepCheckDownloadable() NN_NOEXCEPT;
    nn::Result StepDownloadFiles(nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT;
    //
    nn::Result GetIncompleteFileIndex(int* outIndex) NN_NOEXCEPT;
    nn::Result GetLastProgress() NN_NOEXCEPT;
    //
    nn::Result DownloadFile(const ListFile& file, nn::nifm::NetworkConnection* pConnection, void* buffer, size_t bufferSize) NN_NOEXCEPT;
    //
    bool IsCanceled() NN_NOEXCEPT;
};

}}}}}
