﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      エントリーの差分のビューです。

    @details
                ローカルのエントリーリストと配信リスト内のエントリーリストを比較し、追加・削除・更新等のアクションを決定します。@n
                更新判定にはダイジェスト値を利用します。
*/
template <typename EntryMetaView, typename ListEntry, size_t MaxCount> class DiffView
{
public:
    /*!
        @brief      コンストラクタです。
    */
    DiffView() NN_NOEXCEPT :
        m_Mutex(true),
        m_LocalCount(0),
        m_RemoteCount(0)
    {
    }

    /*!
        @brief      代入処理を行います。

        @param[in]  obj 代入元。

        @return     本オブジェクトの参照。
    */
    DiffView& operator=(const DiffView& obj) NN_NOEXCEPT
    {
        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        for (int i = 0; i < obj.m_LocalCount; i++)
        {
            m_LocalList[i] = obj.m_LocalList[i];
        }

        m_LocalCount = obj.m_LocalCount;

        for (int i = 0; i < obj.m_RemoteCount; i++)
        {
            m_RemoteList[i] = obj.m_RemoteList[i];
        }

        m_RemoteCount = obj.m_RemoteCount;

        return *this;
    }

    /*!
        @brief      差分リストを作成します。

        @param[in]  localView   ローカルのメタ情報ビュー。
        @param[in]  remoteList  配信リスト内のエントリーリスト。
        @param[in]  remoteCount 配信リスト内のエントリー数。

        @return     処理結果。

        @pre
            - remoteList != nullptr
    */
    nn::Result Make(const EntryMetaView& localView, const ListEntry* remoteList, int remoteCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(remoteList);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        int localCount = localView.GetCount();

        for (int i = 0; i < localCount; i++)
        {
            m_LocalList[i] = DiffType_Remove;
        }

        int removeCount = localCount;

        for (int i = 0; i < remoteCount; i++)
        {
            typename EntryMetaView::Entry entry = {};
            int index = 0;

            if (localView.Search(&index, &entry, remoteList[i].name.value).IsSuccess())
            {
                if (std::memcmp(&entry.digest.value, &remoteList[i].digest.value, sizeof (entry.digest.value)) == 0)
                {
                    m_LocalList[index] = DiffType_NotModified;
                    m_RemoteList[i] = DiffType_NotModified;
                }
                else
                {
                    m_LocalList[index] = DiffType_Update;
                    m_RemoteList[i] = DiffType_Update;
                }

                removeCount--;
            }
            else
            {
                m_RemoteList[i] = DiffType_Add;
            }
        }

        NN_RESULT_THROW_UNLESS(remoteCount - removeCount <= DeliveryCacheFileCountMaxPerDirectory, ResultEntryLimitReached());

        m_LocalCount = localCount;
        m_RemoteCount = remoteCount;

        return nn::ResultSuccess();
    }

    /*!
        @brief      指定した差分種別のインデックスリストを取得します。（ローカル）

        @param[out] outCount    取得したインデックス数。
        @param[out] outIndexes  インデックスリスト。
        @param[in]  count       インデックスリストの要素数。
        @param[in]  filter      差分種別。

        @pre
            - outCount != nullptr
            - outIndexes != nullptr
            - count > 0

        @details
                    指定した差分種別と一致するアクションが発生するエントリーのインデックスを取得します。@n
                    インデックスは、 @ref DiffView::Make に指定したローカルのメタ情報ビューのものになります。
    */
    void GetIndexListFromLocal(int* outCount, int* outIndexes, int count, DiffType filter) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outCount);
        NN_SDK_REQUIRES_NOT_NULL(outIndexes);
        NN_SDK_REQUIRES_GREATER(count, 0);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        int actualCount = 0;

        for (int i = 0; i < m_LocalCount; i++)
        {
            if (m_LocalList[i] == filter)
            {
                if (count-- == 0)
                {
                    break;
                }
                outIndexes[actualCount++] = i;
            }
        }

        *outCount = actualCount;
    }

    /*!
        @brief      指定した差分種別のインデックスリストを取得します。（配信リスト）

        @param[out] outCount    取得したインデックス数。
        @param[out] outIndexes  インデックスリスト。
        @param[in]  count       インデックスリストの要素数。
        @param[in]  filter      差分種別。

        @pre
            - outCount != nullptr
            - outIndexes != nullptr
            - count > 0

        @details
                    指定した差分種別と一致するアクションが発生するエントリーのインデックスを取得します。@n
                    インデックスは、 @ref DiffView::Make に指定した配信リストのものになります。
    */
    void GetIndexListFromRemote(int* outCount, int* outIndexes, int count, DiffType filter) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outCount);
        NN_SDK_REQUIRES_NOT_NULL(outIndexes);
        NN_SDK_REQUIRES_GREATER(count, 0);

        std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

        int actualCount = 0;

        for (int i = 0; i < m_RemoteCount; i++)
        {
            if (m_RemoteList[i] == filter)
            {
                if (count-- == 0)
                {
                    break;
                }
                outIndexes[actualCount++] = i;
            }
        }

        *outCount = actualCount;
    }

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    DiffType m_LocalList[MaxCount];
    int m_LocalCount;
    //
    DiffType m_RemoteList[MaxCount];
    int m_RemoteCount;
};

/*!
    @brief      ファイル差分のビューです。

    @details
                ローカルのファイルリストと配信リスト内のファイルリストを比較し、追加・削除・更新等のアクションを決定します。
*/
class FileDiffView : public DiffView<FileMetaView, ListFile, DeliveryCacheFileCountMaxPerDirectory>
{
};

/*!
    @brief      ディレクトリ差分のビューです。

    @details
                ローカルのディレクトリリストと配信リスト内のディレクトリリストを比較し、追加・削除・更新等のアクションを決定します。
*/
class DirectoryDiffView : public DiffView<DirectoryMetaView, ListDirectory, DeliveryCacheDirectoryCountMax>
{
};

}}}}}
