﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryListReader.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>
#include <nn/bcat/detail/service/core/bcat_DiffView.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

/*!
    @brief      配信リストの差分検出を行うモジュールです。
*/
class DeliveryListSubtractor
{
public:
    /*!
        @brief      コンストラクタです。
    */
    DeliveryListSubtractor() NN_NOEXCEPT;

    /*!
        @brief      ダウンロードと同期の対象となるディレクトリを設定します。

        @param[in]  dirName ディレクトリ名。

        @pre
            - dirName != nullptr

        @details
                    本関数を使用した場合、総ダウンロードサイズの計算は対象となるディレクトリに限定されます。
    */
    void SetTargetDirectory(const char* dirName) NN_NOEXCEPT;

    /*!
        @brief      配信リストの解析を行います。

        @param[in]  appId       アプリケーション ID。
        @param[in]  appVersion  アプリケーションバージョン。
        @param[in]  buffer      ワークバッファ。
        @param[in]  bufferSize  ワークバッファのサイズ。

        @return     処理結果。

        @pre
            - appId != nn::ApplicationId::GetInvalidId()
            - buffer != nullptr
            - bufferSize > 0
    */
    nn::Result Analyze(nn::ApplicationId appId, uint32_t appVersion, void* buffer, size_t bufferSize) NN_NOEXCEPT;

    /*!
        @brief      総ダウンロードサイズを取得します。

        @return     総ダウンロードサイズ。

        @details
                    本関数は、 @ref DeliveryListSubtractor::Analyze 成功後のみ、有効な値が返ります。
    */
    int64_t GetWholeDownloadSize() const NN_NOEXCEPT;

    /*!
        @brief      ディレクトリリストを取得します。

        @param[out] outCount    取得したディレクトリ数。
        @param[in]  dirs        ディレクトリバッファ。
        @param[in]  count       ディレクトリバッファの要素数。

        @pre
            - outCount != nullptr
            - dirs != nullptr
            - count > 0

        @details
                    本関数は、 @ref DeliveryListSubtractor::Analyze 成功後のみ、有効な値が取得できます。
    */
    void GetDirectoryView(DirectoryMetaView* outMetaView, DirectoryDiffView* outDiffView) const NN_NOEXCEPT;

    /*!
        @brief      ディレクトリリストを取得します。

        @param[out] outCount    取得したディレクトリ数。
        @param[in]  dirs        ディレクトリバッファ。
        @param[in]  count       ディレクトリバッファの要素数。

        @pre
            - outCount != nullptr
            - dirs != nullptr
            - count > 0

        @details
                    本関数は、 @ref DeliveryListSubtractor::Analyze 成功後のみ、有効な値が取得できます。
    */
    void GetDirectoryList(int* outCount, ListDirectory* dirs, int count) const NN_NOEXCEPT;

    /*!
        @brief      差分サイズが増加するディレクトリのインデックスリストを取得します。

        @param[out] outCount    取得したインデックス数。
        @param[in]  indexes     インデックスバッファ。
        @param[in]  count       インデックスバッファの要素数。

        @pre
            - outCount != nullptr
            - indexes != nullptr
            - count > 0

        @details
                    本関数は、 @ref DeliveryListSubtractor::Analyze 成功後のみ、有効な値が取得できます。
    */
    void GetIncreasingDirectoryIndexList(int* outCount, int* indexes, int count) const NN_NOEXCEPT;

    /*!
        @brief      差分サイズが減少するディレクトリのインデックスリストを取得します。

        @param[out] outCount    取得したインデックス数。
        @param[in]  indexes     インデックスバッファ。
        @param[in]  count       インデックスバッファの要素数。

        @pre
            - outCount != nullptr
            - indexes != nullptr
            - count > 0

        @details
                    本関数は、 @ref DeliveryListSubtractor::Analyze 成功後のみ、有効な値が取得できます。
    */
    void GetReducingDirectoryIndexList(int* outCount, int* indexes, int count) const NN_NOEXCEPT;

private:
    //
    mutable nn::os::Mutex m_Mutex;
    //
    nn::ApplicationId m_AppId;
    uint32_t m_AppVersion;
    //
    const char* m_TargetDirectory;
    //
    ListDirectory m_Dirs[DeliveryCacheDirectoryCountMax];
    int m_DirCount;
    //
    DirectoryMetaView m_DirectoryMetaView;
    DirectoryDiffView m_DirectoryDiffView;
    //
    DeliveryListReader::WalkWorkBuffer m_WalkWorkBuffer;
    //
    FileMetaView m_FileMetaView;
    FileDiffView m_FileDiffView;
    //
    int64_t m_WholeDownloadSize;
    //
    int m_IncreasingDirIndexes[DeliveryCacheDirectoryCountMax];
    int m_IncreasingDirCount;
    int m_ReducingDirIndexes[DeliveryCacheDirectoryCountMax];
    int m_ReducingDirCount;

private:
    //
    nn::Result StepAnalyze(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    //
    nn::Result AnalyzeCallback(const ListDirectory& dir, const ListFile* files, int count) NN_NOEXCEPT;

private:
    //
    static void AnalyzeCallback(const ListDirectory& dir, const ListFile* files, int count, void* param) NN_NOEXCEPT;
};

}}}}}
