﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/bcat/bcat_TypesAdmin.h>
#include <nn/bcat/bcat_Result.h>
#include <nn/bcat/bcat_DeliveryCacheProgress.h>
#include <nn/nn_ApplicationId.h>

namespace nn { namespace bcat {

/*!
    @name データ配信
    @{
*/

/*!
    @brief      データ配信キャッシュストレージをマウントします。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。
    @retval     ResultStorageNotFound       データ配信キャッシュストレージが存在していない。
    @retval     ResultLocked                データ配信キャッシュストレージはロックされている。
    @retval     ResultServiceUnavailable    データ配信機能が利用可能な状態ではない。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                本関数は、任意のアプリケーションのデータ配信キャッシュストレージをマウントすることができます。

                システムが本関数を呼び出したアプリケーションの配信データをダウンロードしている間、システムがデータ配信キャッシュストレージをロックします。@n
                この場合、本関数は ResultLocked を返します。
*/
nn::Result MountDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @brief      データ配信キャッシュストレージを削除します。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。
    @retval     ResultStorageNotFound   データ配信キャッシュストレージが存在していない。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()
*/
nn::Result DeleteDeliveryCacheStorage(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @brief      データ配信キャッシュの同期を要求します。（即時同期）

    @param[out] outProgress 進捗オブジェクト。
    @param[in]  appId       アプリケーション ID。
    @param[in]  appVersion  アプリケーションバージョン。

    @return     処理結果。

    @pre
        - outProgress != nullptr
        - appId != nn::ApplicationId::GetInvalidId()
        - データ配信キャッシュストレージをマウントしていない。

    @details
                本関数は、任意のアプリケーションのデータ配信キャッシュの同期を要求することができます。

                本関数を呼ぶ前に、インターネット接続要求が受理されている必要があります。@n
                インターネット接続要求が受理されてない場合、 ResultInternetRequestNotAccepted が返ります。
*/
nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress, nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT;

/*!
    @brief      データ配信キャッシュの同期を要求します。（即時同期・ディレクトリ指定）

    @param[out] outProgress 進捗オブジェクト。
    @param[in]  appId       アプリケーション ID。
    @param[in]  appVersion  アプリケーションバージョン。
    @param[in]  dirName     ディレクトリ名。

    @return     処理結果。
    @retval     ResultTooManySyncRequests           同期要求を出しすぎている。
    @retval     ResultInternetRequestNotAccepted    インターネット接続要求が受理されていない。

    @pre
        - outProgress != nullptr
        - appId != nn::ApplicationId::GetInvalidId()
        - dirName != nullptr
        - データ配信キャッシュストレージをマウントしていない。

    @details
                本関数は、任意のアプリケーションのデータ配信キャッシュの同期を要求することができます。

                本関数を呼ぶ前に、インターネット接続要求が受理されている必要があります。@n
                インターネット接続要求が受理されてない場合、 ResultInternetRequestNotAccepted が返ります。
*/
nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT;

/*!
    @}
    @name システム設定
    @{
*/

/*!
    @brief      データの復号鍵を生成するパスフレーズを設定します。

    @param[in]  appId       アプリケーション ID。
    @param[in]  passphrase  パスフレーズ。

    @return     処理結果。
    @retval     ResultStorageNotFound   データ配信キャッシュストレージが存在していない。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()
        - passphrase != nullptr
        - 1 <= strlen(passphrase) <= PassphraseLengthMax

    @details
                データの復号鍵は、アプリケーション ID とパスフレーズの組み合わせによって生成されます。

                本関数で設定したパスフレーズは、アプリケーションのデータ配信キャッシュストレージに保存されます。@n
                そのため、本関数を呼び出す前にデータ配信キャッシュストレージが作成されている必要があります。

                パスフレーズがすでに保存されていた場合、本関数は新しいパスフレーズで上書き保存します。

                一般アプリケーションは @ref Initialize 時に自動的にパスフレーズの設定が行われます。
*/
nn::Result SetPassphrase(nn::ApplicationId appId, const char* passphrase) NN_NOEXCEPT;

/*!
    @brief      バックグラウンドで動作するデータ配信タスクを登録します。

    @param[in]  appId       アプリケーション ID。
    @param[in]  appVersion  アプリケーションバージョン。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()

    @details
                タスクを登録することで、アプリケーションを起動していない間に配信データの更新を検知し、自動的に同期が行われます。

                システムが管理するタスク数には上限があり、上限に達すると古いタスクを自動的に削除します。@n
                本関数を呼び出すことで、指定したアプリケーションのタスクがシステム管理のタスクリストの先頭に移動して削除されにくくなります。

                データ配信機能を利用する際は、アプリケーションの起動毎に本関数を呼び出すことを推奨します。@n
                なお、一般アプリケーションは @ref Initialize 時に自動的にタスクの登録が行われます。
*/
nn::Result RegisterBackgroundDeliveryTask(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT;

/*!
    @brief      バックグラウンドで動作するデータ配信タスクの登録を解除します。

    @param[in]  appId   アプリケーション ID。

    @return     処理結果。

    @pre
        - appId != nn::ApplicationId::GetInvalidId()
*/
nn::Result UnregisterBackgroundDeliveryTask(nn::ApplicationId appId) NN_NOEXCEPT;

/*!
    @}
*/

}}
