﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       オーディオコントローラに関する公開ヘッダファイルです。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/audioctrl/audioctrl_AudioControllerTypes.h>

namespace nn { namespace audioctrl {

/**
 * @brief       指定されたターゲットの音量を返します。
 * @param[in]   target                  音声入出力ターゲット
 * @pre         target は有効な音声入出力ターゲット
 * @return      ターゲットに設定されている音量
 */
int GetTargetVolume(AudioTarget target) NN_NOEXCEPT;

/**
 * @brief       指定されたターゲットの音量を設定します。
 * @param[in]   target                  音声入出力ターゲット
 * @param[in]   volume                  音量
 * @pre
 * - target は 有効な音声入出力ターゲット
 * - volume >= nn::audioctrl::GetTargetVolumeMin() && volume <= nn::audioctrl::GetTargetVolumeMax()
 */
void SetTargetVolume(AudioTarget target, int volume) NN_NOEXCEPT;

/**
 * @brief       nn::audioctrl::SetTargetVolume() で設定できる最大音量を取得します。
 * @return      最大音量を返します。
 */
int GetTargetVolumeMax() NN_NOEXCEPT;

/**
 * @brief       nn::audioctrl::SetTargetVolume() で設定できる最小音量を取得します。
 * @return      最小音量を返します。
 */
int GetTargetVolumeMin() NN_NOEXCEPT;

/**
 * @brief       指定されたターゲットのミュート状態を返します。
 * @param[in]   target                  音声入出力ターゲット
 * @pre         target は有効な音声入出力ターゲット
 * @return      ターゲットに設定されているミュート状態
 */
bool IsTargetMute(AudioTarget target) NN_NOEXCEPT;

/**
 * @brief       指定されたターゲットのミュート状態を設定します。
 * @param[in]   target                  音声入出力ターゲット
 * @param[in]   isMute                  ミュート状態
 * @pre
 * - target は 有効な音声入出力ターゲット
 */
void SetTargetMute(AudioTarget target, bool isMute) NN_NOEXCEPT;

/**
 * @brief       指定された音声入出力ターゲットの接続状態を返します。
 * @param[in]   target                  音声入出力ターゲット
 * @pre         target は有効な音声入出力ターゲット
 * @return      指定された音声入出力ターゲットが接続されているか
*/
bool IsTargetConnected(AudioTarget target) NN_NOEXCEPT;

/**
 * @brief       指定された音声入出力ターゲットのチャンネル設定を本体設定から取得します。
 * @param[in]   target                  音声入出力ターゲット
 * @return      チャンネル設定を返します。
 * @pre         target は有効な音声入出力ターゲット
 */
AudioOutputMode GetOutputModeSetting(AudioTarget target) NN_NOEXCEPT;

/**
 * @brief       指定された音声入出力ターゲットの音声入出力設定を本体設定に設定します。
 * @param[in]   target                  音声入出力ターゲット
 * @param[in]   mode                    音声入出力設定
 * @pre         target は有効な音声入出力ターゲット
 * @post        mode == @ref GetOutputModeSetting ( target )
 */
void SetOutputModeSetting(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT;

/**
 * @brief       指定された音声入出力ターゲットの音声入出力設定をデバイスドライバから取得します。
 * @param[in]   target                  音声入出力ターゲット
 * @return      音声入出力設定を返します。
 * @pre         target は有効な音声入出力ターゲット
 */
AudioOutputMode GetAudioOutputMode(AudioTarget target) NN_NOEXCEPT;

/**
 * @brief       指定された音声入出力ターゲットの音声入出力設定をデバイスドライバに設定します。
 * @param[in]   target                  音声入出力ターゲット
 * @param[in]   mode                    音声入出力設定
 * @pre         target は有効な音声入出力ターゲット
 * @post        mode == @ref GetAudioOutputMode ( target )
 */
void SetAudioOutputMode(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT;

/**
 * @brief       強制ミュート機能のポリシーを設定します。
 * @param[in]   policy    強制ミュート機能のポリシー
 */
void SetForceMutePolicy(ForceMutePolicy policy) NN_NOEXCEPT;

/**
 * @brief       強制ミュート機能のポリシーを取得します。
 * @return      強制ミュート機能のポリシー
 */
ForceMutePolicy GetForceMutePolicy() NN_NOEXCEPT;

/**
 * @brief       ヘッドホン出力レベルの設定を行います。
 * @param[in]   mode    ヘッドホン出力レベルの設定
 * @pre         mode は有効なヘッドホン出力レベルの設定値
 * @post        mode == @ref GetHeadphoneOutputLevelMode()
 */
void SetHeadphoneOutputLevelMode(HeadphoneOutputLevelMode mode) NN_NOEXCEPT;

/**
 * @brief       ヘッドホン出力レベルの設定を取得します。
 * @return      ヘッドホン出力レベルの設定
 */
HeadphoneOutputLevelMode GetHeadphoneOutputLevelMode() NN_NOEXCEPT;

/**
 * @brief       ヘッドホン音量についての警告を表示したことをシステムに通知します。
 */
void NotifyHeadphoneVolumeWarningDisplayedEvent() NN_NOEXCEPT;

/**
 * @brief       システムの出力マスターボリュームを取得します。
 * @pre
 * - pOutVolume != nullptr
 * @return      システムの出力マスターボリューム
 */
float GetSystemOutputMasterVolume() NN_NOEXCEPT;

/**
 * @brief       システムの出力マスターボリュームを設定します。
 * @param[in]   volume                  ボリューム
 * @pre
 * - nn::audioctrl::SystemOutputMasterVolumeMin <= volume && volume <= nn::audioctrl::SystemOutputMasterVolumeMax
 */
void SetSystemOutputMasterVolume(float volume) NN_NOEXCEPT;

/**
 * @brief       ヘッドフォン出力音量設定を更新します。
 * @param[in]   isRestricted ペアレンタルコントロールが有効か否か
 */
void UpdateHeadphoneSettings(bool isRestricted) NN_NOEXCEPT;

}}  // namespace nn::audioctrl
