﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/arp/arp_Types.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_ApplicationId.h>
#include <nn/os/os_Types.h>

namespace nn { namespace ns {
struct ApplicationControlProperty;
}} // ~namespace nn::ns

namespace nn { namespace arp { namespace detail {
class IRegistrar;
}}} // ~namespace nn::arp::detail

namespace nn { namespace arp {

/** @brief アプリケーションプロセスに性質を付与するためのクラス
    @details
        状態:
        - 未初期化状態 ... デストラクタ以外のメンバ関数を呼べない。
        - 初期化済状態 ... すべてのメンバ関数を呼べる
*/
class Registrar
{
    friend Result AcquireRegistrar(Registrar*) NN_NOEXCEPT;
    NN_DISALLOW_COPY(Registrar);

private:
    detail::IRegistrar* m_Ptr;

    explicit Registrar(detail::IRegistrar* ptr) NN_NOEXCEPT;
    void Release() NN_NOEXCEPT;

public:
    /** @brief 未初期化状態のオブジェクトを作成するコンストラクタ
    */
    Registrar() NN_NOEXCEPT;
    Registrar(Registrar&& rhs) NN_NOEXCEPT;
    ~Registrar() NN_NOEXCEPT;

    Registrar& operator=(Registrar&& rhs) NN_NOEXCEPT;

    /** @brief ApplicationLaunchProperty を付与する
        @pre
            - *this が初期化済状態
    */
    Result SetApplicationLaunchProperty(const ApplicationLaunchProperty& launchProp) NN_NOEXCEPT;

    /** @brief ApplicationControlProperty を付与する
        @pre
            - *this が初期化済状態
    */
    Result SetApplicationControlProperty(const ns::ApplicationControlProperty& ctrlProp) NN_NOEXCEPT;

    /** @brief 性質の付与を確定する
        @pre
            - *this が初期化済状態
        @post
            - *this が未初期化状態
    */
    Result Issue(const os::ProcessId& pid) NN_NOEXCEPT;
};

/** @brief アプリケーションプロセスに性質を付与するための Registrar オブジェクトを初期化します。
    @post
        - *pRegistrar が初期化済み状態
    @details
        この API を使用するには arp:w 権限が必要です。
*/
Result AcquireRegistrar(Registrar* pRegistrar) NN_NOEXCEPT;

/** @brief 指定した ProcessId をもつプロセスに付与された性質を削除します。
    @details
        この API を使用するには arp:w 権限が必要です。
*/
void DeleteProperties(const os::ProcessId& pid) NN_NOEXCEPT;

/** @brief 指定した ProcessId をもつプロセスに付与された性質を取得します。
    @details
        この API を使用するには arp:r 権限が必要です。
*/
Result GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const os::ProcessId& pid) NN_NOEXCEPT;

/** @brief 指定した ApplicationId をもつプロセスに付与された性質を取得します。
    @details
        この API を使用するには arp:r 権限が必要です。
    @deprecated
        本関数は、特殊な用途（バックグラウンドで動作する処理でアプリケーションの ProcessId が取得できない場合等）での利用を想定しています。@n
        通常は、ProcessId を引数に取る API を使用してください。

        本関数を利用する場合、明示的に警告を抑制してください。
*/
NN_DEPRECATED Result GetApplicationLaunchProperty(ApplicationLaunchProperty* pOut, const ApplicationId& applicationId) NN_NOEXCEPT;

/** @brief 指定した ProcessId をもつプロセスに付与された性質を取得します。
    @details
        この API を使用するには arp:r 権限が必要です。
*/
Result GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const os::ProcessId& pid) NN_NOEXCEPT;

/** @brief 指定した ApplicationId をもつプロセスに付与された性質を取得します。
    @details
        この API を使用するには arp:r 権限が必要です。
    @deprecated
        本関数は、特殊な用途（バックグラウンドで動作する処理でアプリケーションの ProcessId が取得できない場合等）での利用を想定しています。@n
        通常は、ProcessId を引数に取る API を使用してください。

        本関数を利用する場合、明示的に警告を抑制してください。
*/
NN_DEPRECATED Result GetApplicationControlProperty(ns::ApplicationControlProperty* pOut, const ApplicationId& applicationId) NN_NOEXCEPT;

}} // ~namespace nn::arp
