﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <memory>
#include <utility>

namespace nn { namespace am { namespace utility {

void* AllocateStaticDefault(std::size_t byteSize) NN_NOEXCEPT;
void DeallocateStaticDefault(void* p, std::size_t byteSize) NN_NOEXCEPT;

template <typename Tag>
inline void* AllocateStatic(std::size_t byteSize) NN_NOEXCEPT
{
    return AllocateStaticDefault(byteSize);
}

template <typename Tag>
inline void DeallocateStatic(void* p, std::size_t byteSize) NN_NOEXCEPT
{
    return DeallocateStaticDefault(p, byteSize);
}

template <typename T, typename Tag>
class StaticAllocatorWithTag
{
public:
    typedef T value_type;
    value_type* allocate(std::size_t n) NN_NOEXCEPT
    {
        return static_cast<value_type*>(AllocateStatic<Tag>(sizeof(T) * n));
    }
    void deallocate(value_type* p, std::size_t n) NN_NOEXCEPT
    {
        DeallocateStatic<Tag>(static_cast<void*>(p), sizeof(T) * n);
    }
    StaticAllocatorWithTag() {}
    template <typename U>
    StaticAllocatorWithTag(const StaticAllocatorWithTag<U, Tag>&) NN_NOEXCEPT {}
    template <typename U>
    bool operator==(const StaticAllocatorWithTag<U, Tag>&) const NN_NOEXCEPT
    {
        return true;
    }
    template <typename U>
    bool operator!=(const StaticAllocatorWithTag<U, Tag>&) const NN_NOEXCEPT
    {
        return false;
    }
};

template <typename Tag, typename T, typename... Args>
std::shared_ptr<T> MakeSharedWithTag(Args&&... args) NN_NOEXCEPT
{
    return std::allocate_shared<T>(StaticAllocatorWithTag<T, Tag>(), std::forward<Args>(args)...);
};

}}}
