﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <memory>
#include <nn/am/service/window/am_WindowControllerBase.h>

#include <nn/am/service/window/am_AudioWindowController.h>
#include <nn/am/service/window/am_ForegroundWindowController.h>
#include <nn/am/service/window/am_HomeButtonWindowController.h>
#include <nn/am/service/window/am_PowerButtonWindowController.h>
#include <nn/am/service/window/am_CaptureButtonWindowController.h>
#include <nn/am/service/window/am_SleepWindowController.h>
#include <nn/am/service/window/am_NfcWindowController.h>
#include <nn/am/service/window/am_HidWindowController.h>
#include <nn/am/service/window/am_ApmEventWindowController.h>
#include <nn/am/service/window/am_OmmEventWindowController.h>
#include <nn/am/service/window/am_IdleWindowController.h>
#include <nn/am/service/window/am_SpsmEventWindowController.h>
#include <nn/am/service/window/am_DisplayWindowController.h>
#include <nn/am/service/window/am_CecWindowController.h>
#include <nn/am/service/window/am_SdCardEventWindowController.h>
#include <nn/am/service/window/am_GameCardEventWindowController.h>

namespace nn { namespace am { namespace service { namespace window {

typedef detail::IntegratedWindowController<detail::DummyWindowController
    // コントローラを追加する場合はここに列挙する
    , ApmEventWindowController
    , AudioWindowController
    , ForegroundWindowController
    , HomeButtonWindowController
    , PowerButtonWindowController
    , CaptureButtonWindowController
    , OmmEventWindowController
    , IdleWindowController
    , SdCardEventWindowController
    , SleepWindowController
    , NfcWindowController
    , HidWindowController
    , SpsmEventWindowController
    , DisplayWindowController
    , CecWindowController
    , GameCardEventWindowController
> WindowController;

typedef WindowController::Status WindowStatus;
typedef WindowController::EventStatus WindowEventStatus;

struct WindowProperty
    : public CommonWindowProperty
    , public WindowController::Property
{
};

}}}}

#if 0

// WindowController の詳細

// SampleWindowController.h

#include <nn/util/util_Exchange.h>

namespace nn { namespace am { namespace service { namespace window {

class SampleWindowController
    : public WindowControllerBase<SampleWindowController>
    // WindowControllerBase に自身の型を渡して public 継承する
{
public:

    /*
        プロパティ型の定義(必須)
          Property という名前の型を定義し、内部にウインドウの保持する「プロパティ」を記述する。
          プロパティは、Window の性質を表したものであり、外部からこの値が変更することができる。
          プロパティの変更に応じて、ステータス(後述)が適切に計算されウインドウの状態が変化する。
    */
    struct Property
    {
        float sampleValue; // サンプル値
        float sampleValueRate; // サンプル値レート
        bool requestsToHandleSampleMessage; // メッセージハンドル「したいか」どうか
    };

    /*
        ステータス型の定義(必須)
          Status という名前の型を定義し、内部にウインドウの「ステータス」を記述する。
          ステータスは、Window の現状態を表したものであり、
          その他のウインドウや、ウインドウのステータスから、計算される。
          ステータスの計算は、後述の CalculateStatus() 関数で行われる。

          一般に、ウインドウが単独で存在したり、ほぼ最上位のオーダーに存在する場合には、
          ウインドウのプロパティの値が、対応するステータスの値にそのまま反映されることが多い。
          しかし、そうでないような場合には、その他のウインドウの影響を受け、
          プロパティの値と、対応するステータスの値とが、大きく違う値になることがある。
    */
    struct Status
    {
        float effectiveSampleValue; // 実際のサンプル値
        bool handlesSampleMessage; // メッセージをハンドル「するか」どうか
    };

    /*
        各ウインドウのステータス計算の詳細

        - アクティブでないウインドウのステータス status は SetNotActive(&status) によって計算する。
        - アクティブなウインドウのステータスは以下のように計算される。
          1. 計算コンテキスト context を InitializeCalculationContext(&conetxt) で初期化する
          2. ウインドウのリストを、ウインドウのオーダーによってソートする。(オーダーが高いほうが前)
          3. ソートされたウインドウに対し順に以下を実行する。
             - ウインドウのステータス status を CalculateStatus(&status, commonProp, prop, &context) で計算する。
               - commonProp: ウインドウの共通プロパティ
               - prop: ウインドウのプロパティ
    */

    /*
        ウインドウがアクティブでないときのステータス設定関数の定義(必須)
    */
    static void SetNotActive(Status* pStatus) NN_NOEXCEPT
    {
        pStatus->effectiveSampleValue = 0.f;
        pStatus->handlesSampleMessage = false;
    }

    /*
        ステータス計算コンテキストの定義(必須)
    */
    struct CalculationContext
    {
        float currentSampleValueRate; // 現在のサンプル値レート
        bool rightToHandleSampleMessage;
    };

    /*
        ステータス計算コンテキスト初期化関数の定義(必須)
    */
    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->currentSampleValueRate = 1.;
        pContext->rightToHandleSampleMessage = true;
    }

    /*
        ステータス計算関数の定義(必須)
    */
    void CalculateStatus(Status* pStatus, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        NN_UNUSED(commonProp); // 多くの場合 commonProp は使用しない

        // サンプル値の計算
        pStatus->effectiveSampleValue = prop.sampleValue * pContext->currentSampleValueRate;

        // コンテキストの更新
        pContext->currentSampleValueRate *= prop.sampleValueRate;

        // サンプルメッセージハンドリング可否の計算とコンテキスト更新
        pStatus->handlesSampleMessage = prop.requestsToHandleSampleMessage && util::Exchange(&pContext->rightToHandleSampleMessage, false);
        #if 0 // 以下と同義
        pStatus->handlesSampleMessage = prop.requestsToHandleSampleMessage && pContext->rightToHandleSampleMessage;
        if (pStatus->handlesSampleMessage)
        {
            pContext->rightToHandleSampleMessage = false;
        }
        #endif
    }

    /*
        Status のアップデートによる実動作

        - Update 系関数群の引数
          - pWindow: 対象となるウインドウ
            - pWindow->GetTransiter(): 対象のウインドウの遷移器
            - static_cast<EventStatus&>(pWindow->RefEventStatus()): 対象のウインドウのイベントステータス(後述)
          - oldStatus: アップレート前のステータス
          - newStatus: アップレート後のステータス
        - Status の変更に応じて、全ウインドウに対し Update 関数群が呼ばれる。
          - 呼び出すかどうかは、ウインドウがアクティブであるかには依らない。
          - 同名の Update 関数は全てのウインドウの全ての WindowController に対して呼ばれたのち、次の Update 関数が呼ばれる。
            - 順番は UpdateNegativeImmediately, UpdateNegative, UpdateImmediately, UpdatePositive, UpdatePositiveImmediately
            - AreEquals(oldStatus, newStatus) に該当する WindowController に対しては呼ばれない。
    */

    /*
        Status の比較関数(オプション)
          この関数が true を返す場合は Update 系関数群が呼ばれない。
          定義を省略した場合には、必ず false が返るものとみなされる。
    */
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.effectiveSampleValue == y.effectiveSampleValue
            && x.handlesSampleMessage == y.handlesSampleMessage;
    }

    /*
        Update 系関数群の定義
          以下の通りに定義する。
          - Negative 系の Update 関数は、主に oldStatus から newStatus への変更によって、ウインドウの機能を減少させる処理を行う。
          - Positive 系の Update 関数は、主に oldStatus から newStatus への変更によって、ウインドウの機能を増加させる処理を行う。
          - 非 Immediate 系の Update 関数は、主に遷移器の呼び出しを行う。
            - 有限時間ブロックしてもよい。
          - Immediately 系の Update 関数は、主にイベントステータスの変更を行う。
            - ブロックしてはいけない。
            - ウインドウシステムのイベント処理関数と排他される。(安全にイベントステータスの変更が行える)
          - 何も処理を行わない関数は定義しなくてよい。空定義とみなされる。
    */

    //static void UpdateNegativeImmediately(Window* pWindow, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

    static void UpdateNegative(Window* pWindow, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT
    {
        if (oldStatus.effectiveSampleValue > newStatus.effectiveSampleValue)
        {
            // newStatus に対する実際の変更処理
        }
    }

    static void UpdateImmediately(Window* pWindow, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT
    {
        // EventStatus を設定
        pWindow->RefEventStatus().handlesSampleMessage = newStatus.handlesSampleMessage;
    }

    static void UpdatePositive(Window* pWindow, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT
    {
        if (oldStatus.effectiveSampleValue < newStatus.effectiveSampleValue)
        {
            // newStatus に対する実際の変更処理
        }
    }

    //static void UpdatePositiveImmediately(Window* pWindow, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

    /*
        イベントハンドリングの詳細

          各ウインドウは、各 WindowController の EventStatus の情報を持ち、
          この値に従って実際のイベントハンドリングを行う。
          EventStatus は、Immediate 系の Update 系関数で変更されることを想定している。

          何らかのイベントが発生しすると WindowManager::NotifyEvent 関数テンプレートが呼ばれる。
          この関数テンプレートは WindowController をテンプレート引数に持ち、
          const WindowController::EventType& 型の引数 e を取る。
          これが呼ばれると、全ての各ウインドウ pWindow に対し、
          WindowController::HandleEvent(pWindow, pWindow->RefEventStatus(), e) が実行される。
          なお、この実行は、Immediate 系の Update 系関数の実行とは排他される。

          WindowController::HandleEvent では、e と eventStatus の内容に従って、
          pWindow->GetTransiter() などを適切に呼び出すことが期待される。
          一般に、eventStatus には、そのウインドウがイベントを処理するべきかどうかなどが入っているため、
          処理を行うべきかどうかなどを決定する。
          pWindow->RefEventStatus() を直接参照してはいけない。
          長時間ブロックしてはいけない。
    */

    /*
        イベントステータスの定義(オプション)
    */
    struct EventStatus
    {
        bool handlesSampleMessage;
    };

    /*
        イベント型の定義(オプション)
    */
    struct EventType
    {
        // 今回はイベントが一つしかないため、不要
    };

    /*
        イベントハンドラ関数の定義(オプション)
    */
    void HandleEvent(Window* pWindow, const EventStatus& eventStatus, const EventType& e) NN_NOEXCEPT
    {
        NN_UNUSED(e);
        if (eventStatus.handlesSampleMessage)
        {
            // 実際の処理
        }
    }

};

}}}}
#endif
