﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/am/service/window/am_WindowControllerBase.h>

#include<nn/util/util_Exchange.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_Optional.h>
#include <nn/nn_TimeSpan.h>
#include <algorithm>

namespace nn { namespace am { namespace service { namespace window {

struct HomeButtonWindowController
     : public WindowControllerBase<HomeButtonWindowController>
{

    // Status という名前でデータ構造を定義
    struct Status
    {
        bool handlesShort;
        bool handlesLong;
        util::optional<TimeSpan> longPressingTimeToSet;
    };

    // active でないときの Status を初期化する関数
    static void SetNotActive(Status* p) NN_NOEXCEPT
    {
        p->handlesShort = false;
        p->handlesLong = false;
        p->longPressingTimeToSet = util::nullopt;
    }

    // これが true を返す場合は Update 系関数は呼ばれない
    static bool AreEqual(const Status& x, const Status& y) NN_NOEXCEPT
    {
        return true
            && x.handlesShort == y.handlesShort
            && x.handlesLong == y.handlesLong
            && x.longPressingTimeToSet == y.longPressingTimeToSet
        ;
    }

    struct Property
    {
        bool handlesHomeButtonShortPressed;
        bool handlesHomeButtonLongPressed;
        bool setsHomeButtonLongPressingTime;
        bool determinesHomeButtonLongPressingTime;
        TimeSpan homeButtonLongPressingTimeToSet = 0; // デフォルト値でキャップされるため、デフォルトを使いたい場合は 0 指定
        TimeSpan defaultHomeButtonLongPressingTimeToSet; // setsHomeButtonLongPressingTime=true のとき専用
    };

    struct CalculationContext
    {
        bool canHandleShort;
        bool canHandleLong;
        util::optional<TimeSpan> longPressingTimeToSet;
    };

    void InitializeCalculationContext(CalculationContext* pContext) NN_NOEXCEPT
    {
        pContext->canHandleShort = true;
        pContext->canHandleLong = true;
        pContext->longPressingTimeToSet = util::nullopt;
    }

    void CalculateStatus(Status* pOut, const CommonWindowProperty& commonProp, const Property& prop, CalculationContext* pContext) NN_NOEXCEPT
    {
        NN_UNUSED(commonProp);
        pOut->handlesShort = prop.handlesHomeButtonShortPressed && util::Exchange(&pContext->canHandleShort, false);
        pOut->handlesLong = prop.handlesHomeButtonLongPressed && util::Exchange(&pContext->canHandleLong, false);
        if (!pContext->longPressingTimeToSet)
        {
            if (prop.determinesHomeButtonLongPressingTime)
            {
                pContext->longPressingTimeToSet = prop.homeButtonLongPressingTimeToSet;
            }
        }
        if (prop.setsHomeButtonLongPressingTime)
        {
            // pContext->longPressingTimeToSet に設定された値をセットする。
            // ただし pContext->longPressingTimeToSet が未設定(nullopt)の場合、0 扱い。
            // さらにその上で prop.defaultHomeButtonLongPressingTimeToSet 以上でキャップする。
            pOut->longPressingTimeToSet = std::max(pContext->longPressingTimeToSet.value_or(TimeSpan(0)), prop.defaultHomeButtonLongPressingTimeToSet);
        }
    }

    // 必要なものを定義(不要なものはコメントアウトすること)
    //static void UpdateNegativeImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateNegative(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    static void UpdateImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositive(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;
    //static void UpdatePositiveImmediately(Window* p, const Status& oldStatus, const Status& newStatus) NN_NOEXCEPT;

    typedef Status EventStatus;

    struct EventType
    {
        bool onPressedShort;
        bool onPressedLong;
    };

    void HandleEvent(Window* p, const EventStatus& eventStatus, const EventType& e) NN_NOEXCEPT;

};

}}}}
