﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/vi/fbshare/vi_SharedMemoryPoolLayout.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/am/service/display/am_SystemSharedLowLayer.h>

namespace nn{ namespace am{ namespace service{ namespace display{ namespace detail{

    class ExtraSharedLayerResourceAccessor;

    // Initialize 後、以下の呼出が可能
    // - Activate
    // - RequestShow
    // - CreateLayer
    //
    // Activate, Deactivate は他と独立して呼び出せる。
    // DestroyLayer により RequestShow が取り消される。
    //
    // 更に CreateLayer の後、 SetTexture が呼び出せる。
    //
    // 以下がすべて満たされたときにディスプレイに送られる
    // - IsActive                (Activate() の呼出)
    // - IsPresentationRequested (RequestShow() の呼出)
    // - HasImage                (SetTexture() の呼出)
    //
    // NOTE:
    //   CreateLayer の前に RequestShow の呼出を行った場合、
    //   CreateLayer -> SetTexture で画像を設定した際にディスプレイに送られる。
    //
    class ExtraSharedLayerManager
    {
    public:
        bool IsInitialized() const NN_NOEXCEPT { return m_pResource != nullptr; }
        bool IsLayerCreated() const NN_NOEXCEPT { return m_IsLayerCreated; }
        bool IsActive() const NN_NOEXCEPT { return m_IsActive; }
        bool IsPresentationRequested() const NN_NOEXCEPT { return m_RequestedTextureIndex >= 0; }
        int GetRequestedTextureIndex() const NN_NOEXCEPT { return m_RequestedTextureIndex; }
        int GetPresentedTextureIndex() const NN_NOEXCEPT { return m_PresentedTextureIndex; }
        int GetBufferWidth(int index)  const NN_NOEXCEPT { return m_Layout.entries[index].width; }
        int GetBufferHeight(int index) const NN_NOEXCEPT { return m_Layout.entries[index].height; }
        int GetImageRegionX(int index) const NN_NOEXCEPT { return m_ImageRegionList[index].m_X; }
        int GetImageRegionY(int index) const NN_NOEXCEPT { return m_ImageRegionList[index].m_Y; }
        int GetImageRegionWidth(int index) const NN_NOEXCEPT { return m_ImageRegionList[index].m_Width; }
        int GetImageRegionHeight(int index) const NN_NOEXCEPT { return m_ImageRegionList[index].m_Height; }
        // 画像が設定されているかを取得します。
        // @return width > 0 && height > 0
        bool HasImage(int index) const NN_NOEXCEPT
        {
            return GetImageRegionWidth(index) > 0 && GetImageRegionHeight(index) > 0;
        }

        void Initialize(const ExtraSharedLayerResourceAccessor& resAccessor) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        // @pre IsInitialized()
        void Activate() NN_NOEXCEPT;

        // @pre IsInitialized()
        void Deactivate() NN_NOEXCEPT;

        // 画像を表示するように要求します。
        // @pre IsInitialized()
        void RequestShow(int index) NN_NOEXCEPT;

        // 画像を非表示にするように要求します。
        // @pre IsInitialized()
        void RequestHide() NN_NOEXCEPT;

        nn::Result CreateLayer(
            nn::sf::NativeHandle&& hTransferMemory,
            size_t transferMemorySize,
            const nn::vi::fbshare::SharedMemoryPoolLayout& layout,
            int zOrder,
            nn::vi::LayerStackFlagType stacks
        ) NN_NOEXCEPT;

        void DestroyLayer() NN_NOEXCEPT;

        // 画像を設定します。
        // 表示中の画像を書き換える場合には一時領域を経由するバージョンを使用してテアリングを回避します。
        // @pre IsInitialized()
        // @pre IsLayerCreated()
        nn::Result SetTexture(int index, int x, int y, int width, int height, const void* buffer, size_t size, int originMode) NN_NOEXCEPT;

        // システム共有バッファを一時領域として使用して画像を設定します。
        // 表示中の画像を書き換える際のテアリングを回避するために使用します。
        // @pre IsInitialized()
        // @pre IsLayerCreated()
        nn::Result SetTextureThroughTempTexture(int index, int x, int y, int width, int height, const void* buffer, size_t size, int originMode, int temporalSystemTextureIndex) NN_NOEXCEPT;

    private:
        void UpdateVisibilityImpl() NN_NOEXCEPT;
        void ForceHideImpl() NN_NOEXCEPT;


    private:
        ExtraSharedLayerResourceAccessor * m_pResource = nullptr;
        NN_ALIGNAS(8) char m_ResourceAccessorStorage[8] = {};

        bool m_IsActive             = false;

        int m_RequestedTextureIndex = -1;

        bool m_IsLayerCreated       = false;
        nn::vi::fbshare::SharedMemoryPoolLayout m_Layout = {};
        nn::vi::fbshare::SharedBufferHandle     m_BufferHandle = {};
        SystemSharedLowLayer m_LowLayer;
        nn::vi::LayerStackFlagType m_LayerStacks = 0;
        struct
        {
            int m_X      = 0;
            int m_Y      = 0;
            int m_Width  = 0;
            int m_Height = 0;
        } m_ImageRegionList[nn::vi::fbshare::SharedBufferTextureCountMax] = {};

        int m_PresentedTextureIndex = -1;

    };

}}}}}
