﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include <nn/vi/fbshare/vi_SharedBufferHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerTextureIndexList.h>
#include <nn/vi/fbshare/vi_SharedMemoryPoolLayout.h>
#include <nn/vi/fbshare/vi_SharedTextureOption.h>
#include <nn/applet/applet_FundamentalTypes.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    class DisplayPrimitiveOperation
    {
    public:
        static void Initialize() NN_NOEXCEPT;
        static void Finalize() NN_NOEXCEPT;

    //@name Display
    //@{
        static nn::Result OpenDisplay(nn::vi::DisplayId* pOutValue, const char* name) NN_NOEXCEPT;
        static void CloseDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT;

        static nn::Result GetDisplayHotplugState(nn::vi::HotplugStateType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        static nn::Result GetDisplayHotplugEvent(nn::sf::NativeHandle& outHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        static nn::Result SetDisplayCmuLuma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT;
        static nn::Result SetDisplayAlpha(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT;
        static nn::Result SetDisplayPowerState(nn::vi::DisplayId displayId, nn::vi::PowerStateType value) NN_NOEXCEPT;
        static nn::Result GetDisplayResolution(int64_t* pOutWidth, int64_t* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT;
        static nn::Result SetDefaultDisplay(nn::vi::DisplayId displayId) NN_NOEXCEPT;
    //@}

    //@name DisplayLayer
    //@{

        static nn::Result CreateDisplayLayer(nn::vi::LayerId* pOutValue, nn::vi::LayerSettings settings, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT;
        static void DestroyDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT;

        static nn::Result AttachDisplayLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static void DetachDisplayLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result StartDisplayLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result StopDisplayLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result StartDisplayLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result StopDisplayLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result InitializeDisplayLayerPresentationAllFencesExpiredEvent(nn::os::SystemEventType* pEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;
        static void FinalizeDisplayLayerPresentationAllFencesExpiredEvent(nn::os::SystemEventType* pEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;

        static void SetDisplayLayerConfig(nn::vi::LayerId layerId, const nn::vi::LayerConfig& config) NN_NOEXCEPT;

        // shared layer 用
        static void SetDisplayLayerZOrder(nn::vi::LayerId layerId, int32_t zOrder) NN_NOEXCEPT;

        // GPU タイムスライスに PresentInterval を反映するレイヤの指定
        static void SetIsConductorLayer(nn::vi::LayerId layerId, bool isConductor) NN_NOEXCEPT;

        static void InitializeLayerTexturePresentingEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;
        static void FinalizeLayerTexturePresentingEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;

    //@}

    //@name IndirectLayer
    //@{
        static nn::Result CreateIndirectLayer(nn::vi::IndirectLayerHandleType* pOutHandle) NN_NOEXCEPT;
        static void DestroyIndirectLayer(nn::vi::IndirectLayerHandleType hLayer) NN_NOEXCEPT;

        static nn::Result CreateIndirectConsumerEndPoint(nn::vi::IndirectConsumerHandleType* pOutHandle, nn::vi::IndirectLayerHandleType hLayer, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT;
        static void DestroyIndirectConsumerEndPoint(nn::vi::IndirectLayerHandleType hLayer, nn::vi::IndirectConsumerHandleType hConsumer) NN_NOEXCEPT;

        static nn::Result CreateIndirectProducerEndPoint(nn::vi::IndirectProducerHandleType* pOutValue, nn::vi::IndirectLayerHandleType hLayer, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT;
        static void DestroyIndirectProducerEndPoint(nn::vi::IndirectLayerHandleType hLayer, nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT;

        // @brief IndirectProducer の Flip タイミングを設定する。
        // @details
        // 初期値は 0。
        // タイミングは VSYNC からのオフセット時間で指定する。 offset が負の場合は 0 として扱われる。
        // offset が 1 フレームの時間よりも長い場合、例えば offset = 25ms (=1.5f) とした場合、 2 フレームに 1 回 Flip される。
        // IndirectProducer の Flip は優先度が高くないため、GPU タイムスライスが切り替わる直前や次の VSYNC の直前に Flip しようとすると Flip 間隔が不安定になる場合がある。
        //
        // Flip タイミングは CreateIndirectLayerProducerEndPoint() の成功から DestroyIndirectLayerProducerEndPoint() の呼出までの間設定可能。
        // ProducerEndPoint を破棄後、再作成した場合には初期値が設定される。クライアントが接続／切断しても設定された値は維持される。
        static void SetIndirectProducerFlipOffset(nn::vi::IndirectLayerHandleType layerHandle, nn::vi::IndirectProducerHandleType producerHandle, nn::TimeSpan offset) NN_NOEXCEPT;
    //@}

    //@name SharedLayer
    //@{

    // SharedBuffer
        static nn::Result CreateSharedBufferOnStaticStorage(nn::vi::fbshare::SharedBufferHandle* pOutHandle, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT;
        static nn::Result CreateSharedBufferOnProcessHeap(nn::vi::fbshare::SharedBufferHandle* pOutHandle, uint64_t hHeapMemory, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT;
        static nn::Result CreateSharedBufferOnTransferMemory(nn::vi::fbshare::SharedBufferHandle* pOutHandle, nn::sf::NativeHandle&& hTransferMemory, size_t size, const nn::vi::fbshare::SharedMemoryPoolLayout& layout) NN_NOEXCEPT;
        static void DestroySharedBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;

        static void RegisterSharedBufferImporter(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        static void UnregisterSharedBufferImporter(nn::vi::fbshare::SharedBufferHandle handle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // 起動ロゴを共有フレームバッファ上に展開する。
        // 起動ロゴは初期状態で共有フレームバッファの先頭から書かれているので先頭のバッファに展開しようとすると絵が壊れる。
        static nn::Result ExpandStartupLogoOnSharedFrameBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer, int dstIndex, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT;

        // 共有バッファ内のフレームバッファ間で画像をコピーする。
        // コピー先の画像に設定されるレイヤスタックは以下のルールで決める
        //   1. コピー元の画像に設定されているレイヤスタックと stacksFilter で指定したレイヤスタックの共通部分を設定する
        //   2. 1. の結果どのレイヤスタックにも入らない場合、 nullStacks で指定したレイヤスタックを設定する
        // （ stacksFilter=0 にすると nullStacks で指定した値をつけることができる）
        static nn::Result CopyDetachedSharedFrameBufferImage(
            nn::vi::fbshare::SharedBufferHandle hDstBuffer,
            int dstIndex,
            nn::vi::fbshare::SharedBufferHandle hSrcBuffer,
            int srcIndex,
            nn::vi::ImageTransform dstTransform,
            nn::vi::fbshare::SharedTextureAlphaOption alphaOption,
            nn::vi::LayerStackFlagType stacksFilter,
            nn::vi::LayerStackFlagType nullStacks
        ) NN_NOEXCEPT;

        static nn::Result SetDetachedSharedFrameBufferImage(
            nn::vi::fbshare::SharedBufferHandle hBuffer,
            int dstIndex,
            const void* srcBuffer,
            size_t srcBufferSize,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransform srcTransform
        ) NN_NOEXCEPT;

        static nn::Result SetDetachedSharedFrameBufferSubImage(
            nn::vi::fbshare::SharedBufferHandle hBuffer,
            int dstIndex,
            int32_t x,
            int32_t y,
            int32_t w,
            int32_t h,
            uint32_t bgColor,
            const void* srcBuffer,
            size_t srcBufferSize,
            const nn::vi::fbshare::SharedTextureOption& dstOption,
            nn::vi::ImageTransform srcTransform
        ) NN_NOEXCEPT;

        static nn::Result FillDetachedSharedFrameBufferColor(nn::vi::fbshare::SharedBufferHandle hBuffer, int dstIndex, uint32_t color, const nn::vi::fbshare::SharedTextureOption& option) NN_NOEXCEPT;
        static nn::Result GetDetachedSharedFrameBufferImage(uint64_t* pOutSize, void* outBuffer, size_t outBufferSize, nn::vi::fbshare::SharedBufferHandle hBuffer, int srcIndex) NN_NOEXCEPT;
        static nn::Result SetDetachedSharedFrameBufferLayerStack(nn::vi::fbshare::SharedBufferHandle hBuffer, int index, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT;

        static nn::Result PresentDetachedSharedFrameBufferToLowLevelLayer(nn::vi::fbshare::SharedBufferHandle hBuffer, nn::vi::LayerId lowLayerId, int index) NN_NOEXCEPT;

    // SharedLayer
        static nn::Result CreateSharedLayer(nn::vi::fbshare::SharedLayerHandle* pOutHandle, nn::applet::AppletResourceUserId userAruid) NN_NOEXCEPT;
        static void DestroySharedLayer(nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;

        static void InitializeSharedLayerDetachReadyEvent(nn::os::SystemEventType* pOutSystemEvent, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;
        static void FinalizeSharedLayerDetachReadyEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::fbshare::SharedLayerHandle h) NN_NOEXCEPT;

        static void SetSharedLayerVisibility(nn::vi::fbshare::SharedLayerHandle hLayer, bool visible, int32_t /*zOrder*/, bool isCaptureTarget) NN_NOEXCEPT;

        static nn::Result AttachSharedLayer(nn::vi::fbshare::SharedLayerHandle handle, nn::vi::LayerId layerId, const nn::vi::fbshare::SharedLayerTextureIndexList& list) NN_NOEXCEPT;
        static void DetachSharedLayer(nn::vi::fbshare::SharedLayerHandle handle, nn::os::SystemEventType* pDetachReadyEvent, nn::TimeSpan timeout) NN_NOEXCEPT;

    // SharedLowLevelLayer
        static void BlankSharedLowLevelLayer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;
        static void SynchronizeSharedLowLevelLayer(int* pOutIndex, nn::vi::LayerId layerId, nn::os::SystemEventType* pSyncEvent) NN_NOEXCEPT;

        static nn::Result BindSharedLowLevelLayerToDisplayLayer(nn::vi::LayerId layerId, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        static void UnbindSharedLowLevelLayerFromDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT;
        static nn::Result ConnectSharedLowLevelLayerToDisplayLayer(nn::vi::LayerId layerId, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;
        static void DisconnectSharedLowLevelLayerFromDisplayLayer(nn::vi::LayerId layerId) NN_NOEXCEPT;

        static nn::Result BindSharedLowLevelLayerToIndirectLayer(nn::vi::IndirectProducerHandleType hProducer, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        static void UnbindSharedLowLevelLayerFromIndirectLayer(nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT;
        static nn::Result ConnectSharedLowLevelLayerToIndirectLayer(nn::vi::IndirectProducerHandleType hProducer, nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT;
        static void DisconnectSharedLowLevelLayerFromIndirectLayer(nn::vi::IndirectProducerHandleType hProducer) NN_NOEXCEPT;

        static void InitializeSharedLowLevelLayerSynchronizedEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;
        static void FinalizeSharedLowLevelLayerSynchronizedEvent(nn::os::SystemEventType* pSystemEvent, nn::vi::LayerId layerId) NN_NOEXCEPT;

    // Other
        static nn::Result AllocateDisplayProcessHeapBlock(uint64_t* pOutHandle, uint64_t size) NN_NOEXCEPT;
        static void FreeDisplayProcessHeapBlock(uint64_t handle) NN_NOEXCEPT;


    //@}

    };

}}}}
