﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <utility>
#include <memory>
#include <list>
#include <array>

#include <nn/applet/applet_Types.h>
#include <nn/am/service/am_IntegratedApplet.h>
#include <nn/am/service/am_Foundation.sfdl.h>
#include <nn/am/service/am_Functions.sfdl.h>
#include <nn/am/service/am_AppletProxy.h>
#include <nn/sf/sf_Types.h>
#include <nn/account/account_Types.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace am { namespace service {

class IntegratedSystemApplet
    : public IntegratedMainApplet
{
private:

    typedef IntegratedMainApplet Derived;

public:

    explicit IntegratedSystemApplet(AppletSystem* pAppletSystem, applet::AppletId appletId) NN_NOEXCEPT;

    void NotifyFloatingApplicationExistence() NN_NOEXCEPT;
    void NotifyApplicationExit() NN_NOEXCEPT;

    NN_AM_SERVICE_DIAGNOSTICS_DEFINE_APPLET_KIND_STRING("SA")

private:

    // 基本情報
    applet::AppletId m_AppletId;

    // overrides(option)
    //virtual Result Before() NN_NOEXCEPT NN_OVERRIDE;
    //virtual Result After() NN_NOEXCEPT NN_OVERRIDE;
    //virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE;
    //virtual void RunBehindProcess(process::NsProcess* pProcess) NN_NOEXCEPT NN_OVERRIDE;
    //virtual void RequestExitOne() NN_NOEXCEPT NN_OVERRIDE;

    // overrides IntegratedApplet
    virtual applet::AppletIdentityInfo GetAppletIdentityInfo() NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result CreateProcessImpl(std::shared_ptr<process::NsProcess>* pOut) NN_NOEXCEPT NN_OVERRIDE final;
    virtual std::shared_ptr<process::NsProcess> MakeAttachedProcess(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE final;

    // overrides IntegratedApplet
    virtual AppletProxyInfo DoCreateAppletProxy(os::ProcessId processId) NN_NOEXCEPT NN_OVERRIDE;

    // for impl
    std::shared_ptr<IntegratedSystemApplet> SharedFromThis() NN_NOEXCEPT
    {
        return std::static_pointer_cast<IntegratedSystemApplet>(this->shared_from_this());
    }

    class SystemAppletSelfProxyImpl;

    // overrieds (window)
    virtual window::WindowProperty GetInitialWindowProperty() NN_NOEXCEPT NN_OVERRIDE
    {
        auto ret = IntegratedApplet::GetInitialWindowProperty();
        ret.handlesHomeButtonShortPressed = true;
        ret.handlesHomeButtonLongPressed = true;
        ret.handlesPowerButtonShortPressed = true;
        ret.handlesPowerButtonLongPressed = true;
        ret.handlesSdCardRemoved = true;
        ret.handlesSleepRequiredByHighTemperature = true;
        ret.handlesSleepRequiredByLowBattery = true;
        ret.handlesAutoPowerDown = true;
        ret.handlesCecSystemStandby = true;
        ret.isMainAppletWindow = true;
        ret.gpuTimeSliceInForeground = GetGpuTimeSliceOf(GpuTimeKind_FgSa);
        ret.gpuTimeSliceInBackground = GetGpuTimeSliceOf(GpuTimeKind_BgSa);
        ret.gpuTimeSlicePriority = 1;
        ret.determinesHomeButtonLongPressingTime = true;
        ret.pWirelessPriorityModeRequest = WirelessPriorityMode_OptimizedForBluetooth;
        return ret;
    }

    virtual GpuResourceGroupId GetGpuResourceGroupId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GpuResourceGroupId_System;
    }

public:

    void PushLaunchRequestedApplication(std::shared_ptr<IntegratedApplication> pApplication) NN_NOEXCEPT;
    std::shared_ptr<IntegratedApplication> PopLaunchRequestedApplication() NN_NOEXCEPT;

private:

    os::Mutex m_LaunchRequestedApplicationListMutex{false};
    std::list<std::shared_ptr<IntegratedApplication>, StaticAllocator<std::shared_ptr<IntegratedApplication>>> m_LaunchRequestedApplicationList;

private:

    virtual void OnHomeButtonLongPressed() NN_NOEXCEPT NN_OVERRIDE;

public:

    // for ISystemAppletControllerForDebug
    Result RequestLaunchApplicationForDebug(ncm::ApplicationId applicationId, util::Span<const account::Uid> uids) NN_NOEXCEPT
    {
        return m_ControllerForDebug.RequestLaunchApplication(applicationId, uids);
    }

private:

    class ControllerForDebug
    {
    public:

        explicit ControllerForDebug(IntegratedSystemApplet* pParent) NN_NOEXCEPT
            : m_pParent(pParent)
        {
        }

        Result RequestLaunchApplication(ncm::ApplicationId applicationId, util::Span<const account::Uid> uids) NN_NOEXCEPT;
        Result PopRequestLaunchApplication(ncm::ApplicationId* pApplicationId, int* pOutUidCount, util::Span<account::Uid> outUids) NN_NOEXCEPT;

    private:

        IntegratedSystemApplet* m_pParent;

        os::SdkMutex m_Mutex;
        auto Lock() NN_NOEXCEPT;

        struct LaunchApplicationRequestInfo
        {
            ncm::ApplicationId applicationId;
            std::array<account::Uid, account::UserCountMax> uids;
            int uidCount;
        };
        util::optional<LaunchApplicationRequestInfo> m_pLaunchApplicationRequestInfo;

    };
    ControllerForDebug m_ControllerForDebug{this};

};

}}}
