﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/am/service/am_Proxies.sfdl.h>

#include <nn/nn_Abort.h>
#include <utility>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/am/am_ResultPrivate.h>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/service/am_ServiceDiagnostics.h>

namespace nn { namespace am { namespace service {

/**
    @brief AppletSystem に登録されるアプレットプロキシの基底クラス

    @details
     仮想関数としては全機能が用意されているが、対象となるアプレットの種類によって、実際にサポートされるかどうかは異なる。
*/
class AppletProxy
{
public:

    // for Common(self)
    virtual nn::Result GetCommonStateGetter(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ICommonStateGetter>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetSelfController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ISelfController>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetAudioController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IAudioController>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetDisplayController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IDisplayController>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetWindowController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IWindowController>> pOut) NN_NOEXCEPT;

    // for Stackable
    virtual nn::Result GetProcessWindingController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IProcessWindingController>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetLibraryAppletCreator(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletCreator>> pOut) NN_NOEXCEPT;

    // for Application only
    virtual nn::Result GetApplicationFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationFunctions>> pOut) NN_NOEXCEPT;

    // for System Applet
    virtual nn::Result GetHomeMenuFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IHomeMenuFunctions>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetGlobalStateController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IGlobalStateController>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetApplicationCreator(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationCreator>> pOut) NN_NOEXCEPT;

    // for Library Applet
    virtual nn::Result OpenLibraryAppletSelfAccessor(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletSelfAccessor>> pOut) NN_NOEXCEPT;

    // for Overlay Applet
    virtual nn::Result GetOverlayFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IOverlayFunctions>> pOut) NN_NOEXCEPT;
    virtual nn::Result GetDebugFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IDebugFunctions>> pOut) NN_NOEXCEPT;

protected:
    virtual ~AppletProxy() NN_NOEXCEPT = 0;
};

template <typename Pointer>
class AppletProxyBase
    : public AppletProxy
{
protected:

    explicit AppletProxyBase(Pointer&& p) NN_NOEXCEPT
        : m_P(std::forward<Pointer>(p))
    {
    }

    const Pointer& GetPointer() const
    {
        return m_P;
    }

    template <typename Interface, typename Impl>
    Result CreateObjectImpl(sf::Out<sf::SharedPointer<Interface>> pOut, Impl&& impl) NN_NOEXCEPT
    {
        auto ret = SfObjectFactory::CreateShared<Interface>(std::forward<Impl>(impl));
        NN_RESULT_THROW_UNLESS(ret, am::ResultOutOfMemory());
        *pOut = std::move(ret);
        NN_RESULT_SUCCESS;
    }

    template <typename Interface, typename Impl>
    Result CreateObjectImpl(sf::Out<sf::SharedPointer<Interface>> pOut, Impl* p) NN_NOEXCEPT
    {
        auto ret = SfObjectFactory::CreateSharedWithoutManagement<Interface>(p);
        NN_RESULT_THROW_UNLESS(ret, am::ResultOutOfMemory());
        *pOut = std::move(ret);
        NN_RESULT_SUCCESS;
    }

private:

    const Pointer m_P;

};

/**
    @brief

    @details
     Pointer::operator-> の返り値は以下を呼べる必要がある。

     - GetCommonStateGetter()
     - GetSelfController()
     - GetWindowController()
     - GetDebugFunctions()

     またそれぞれの返り値は、対応する関数を呼べる必要がある。
*/
template <typename Pointer>
class CommonAppletProxy
    : public AppletProxyBase<Pointer>
{
protected:

    explicit CommonAppletProxy(Pointer&& p) NN_NOEXCEPT
        : AppletProxyBase<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetCommonStateGetter(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ICommonStateGetter>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetCommonStateGetter());
    }

    virtual nn::Result GetSelfController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ISelfController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetSelfController());
    }

    virtual nn::Result GetWindowController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IWindowController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetWindowController());
    }

    virtual nn::Result GetDebugFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IDebugFunctions>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetDebugFunctions());
    }

};

template <typename Pointer>
class StackableAppletProxy
    : public CommonAppletProxy<Pointer>
{
protected:

    explicit StackableAppletProxy(Pointer&& p) NN_NOEXCEPT
        : CommonAppletProxy<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetLibraryAppletCreator(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletCreator>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetLibraryAppletCreator());
    }

    virtual nn::Result GetAudioController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IAudioController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetAudioController());
    }

    virtual nn::Result GetDisplayController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IDisplayController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetDisplayController());
    }

};

template <typename Pointer>
class ApplicationProxy final
    : public StackableAppletProxy<Pointer>
{
public:

    explicit ApplicationProxy(Pointer&& p) NN_NOEXCEPT
        : StackableAppletProxy<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetApplicationFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationFunctions>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetApplicationFunctions());
    }

};

template <typename Pointer>
class SystemAppletProxy final
    : public StackableAppletProxy<Pointer>
{
public:

    explicit SystemAppletProxy(Pointer&& p) NN_NOEXCEPT
        : StackableAppletProxy<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetHomeMenuFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IHomeMenuFunctions>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetHomeMenuFunctions());
    }

    virtual nn::Result GetGlobalStateController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IGlobalStateController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetGlobalStateController());
    }

    virtual nn::Result GetApplicationCreator(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IApplicationCreator>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetApplicationCreator());
    }

};

template <typename Pointer>
class LibraryAppletProxy final
    : public StackableAppletProxy<Pointer>
{
public:

    explicit LibraryAppletProxy(Pointer&& p) NN_NOEXCEPT
        : StackableAppletProxy<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetProcessWindingController(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IProcessWindingController>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetProcessWindingController());
    }

    virtual nn::Result OpenLibraryAppletSelfAccessor(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::ILibraryAppletSelfAccessor>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->OpenLibraryAppletSelfAccessor());
    }

};

template <typename Pointer>
class OverlayAppletProxy final
    : public StackableAppletProxy<Pointer>
{
public:

    explicit OverlayAppletProxy(Pointer&& p) NN_NOEXCEPT
        : StackableAppletProxy<Pointer>(std::forward<Pointer>(p))
    {
    }

private:

    virtual nn::Result GetOverlayFunctions(nn::sf::Out<nn::sf::SharedPointer<nn::am::service::IOverlayFunctions>> pOut) NN_NOEXCEPT NN_OVERRIDE final
    {
        return this->CreateObjectImpl(pOut, this->GetPointer()->GetOverlayFunctions());
    }

};

}}}
