﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>

namespace nn { namespace ae {

/**
 * @brief   動作モードを示す列挙型です。
 *
 */
enum OperationMode
{
    OperationMode_Handheld, //!< 携帯モードです。
    OperationMode_Console,  //!< 据置モードです。
};

/**
 * @brief   現在の動作モードを取得します。
 *
 * @details
 *  動作モードによって、デフォルトの映像・音声出力が変化します。
 *
 *  OperationMode_Handheld の場合は、ユーザは本体 LCD および内蔵スピーカーを通じてゲームをプレイします。
 *  デフォルトの映像出力は本体 LCD になります。
 *  デフォルトの音声出力は本体内蔵スピーカーになります。本体にヘッドフォンが接続されている場合は、ヘッドフォンのみから出力されます。
 *
 *  OperationMode_Console の場合は、ユーザは本体が接続された TV を通じてゲームをプレイします。
 *  デフォルトの映像出力は本体が接続された TV になります。
 *  デフォルトの音声出力は本体が接続された TV になります。本体にヘッドフォンが接続されている場合は、TV およびヘッドフォンから出力されます。
 *
 *  vi ライブラリにおいて、デフォルトの映像出力を使用するには nn::vi::OpenDefaultDisplay() を使用してください。
 *  audio ライブラリにおいて、デフォルトの音声出力を使用するには nn::audio::OpenDefaultAudioOut() を使用してください。
 *  AudioRenderer において、デフォルトの音声出力を使用するには "MainAudioOut" という名前の nn::audio::DeviceSinkType を使用してください。
 *
 *  システムに与える負荷が大きいため頻繁な呼び出しには適していません。
 *  現在の動作モードを監視したい場合は nn::ae::GetNotificationMessage() を使用して、動作モードの変更通知を受け取った際に呼び出してください。
 */
OperationMode GetOperationMode() NN_NOEXCEPT;

/**
* @brief   Switchドック状態を示す列挙型です。
*
*/
enum CradleStatus
{
    CradleStatus_None,            //!< Switchドックに接続されていません。
    CradleStatus_NotEnoughPower,  //!< Switchドックに接続されていますが、給電能力が不十分です。
    CradleStatus_Mountable,       //!< Switchドックに接続されています。
};

/**
* @brief   Switchドック状態を取得します。
*
* @details
*  動作モードの遷移とは非同期的に変化するため、この API を動作モードの判定に用いないでください。
*
*  システムに与える負荷が大きいため頻繁な呼び出しには適していません。
*/
CradleStatus GetCradleStatus() NN_NOEXCEPT;

/**
* @brief    Switchドックのファームウェアバージョンを表す構造体です。
*/
struct CradleFwVersion
{
    uint32_t    pdcH;   //!< PDC-H のファームウェアバージョン。
    uint32_t    pdcA;   //!< PDC-A のファームウェアバージョン。
    uint32_t    mcu;    //!< MCU のファームウェアバージョン。
    uint32_t    dp2hdmi;//!< DP2HDMI のファームウェアバージョン。
};

/**
* @brief      Switchドックのファームウェアバージョンを取得します。
* @param[out] pOutValue Switchドックのファームウェアバージョン。
* @return     ファームウェアバージョンの取得に成功した場合は true、失敗した場合は false。
* @pre
*  - pOutValue != nullptr
*
* @details
*  Switchドックに接続されていない状態で呼び出した場合はファームウェアバージョンの取得に失敗し、false を返します。
*/
bool GetCradleFwVersion(CradleFwVersion* pOutValue) NN_NOEXCEPT;

}} // namespace nn::ae

