﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ライブラリアプレットの制御に関する API の宣言
 *
 * @details
 *  本ヘッダでは、ライブラリアプレットの制御に関する API を宣言しています。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/ae/ae_Types.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/applet/applet_Storage.h>
#include <nn/applet/applet_LibraryApplet.h>
#include <nn/err/err_ErrorContext.h>
#include <nn/err/err_Types.h>
#include <nn/ncm/ncm_StorageId.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/vi/vi_IndirectProducer.private.h>
#include <nn/settings/settings_Language.h>

namespace nn { namespace ae {

//-----------------------------------------------------------------------------

/**
    @brief ライブラリアプレット自身の情報を表す構造体です。
*/
struct LibraryAppletSelfInfo
{
    applet::AppletId appletId; //!< アプレット ID
    applet::LibraryAppletMode libraryAppletMode; //!< ライブラリアプレットモード
    bool isUnwound; //!< 一度終了したものの再起動であるかどうか
    applet::LibraryAppletHandle previousLibraryAppletHandle; //!< 直前のライブラリアプレットハンドル
};

/**
    @brief ストレージハンドを表す型で、applet ライブラリのものと共通で使用できます。
*/
typedef applet::StorageHandle StorageHandle;

//! @name ライブラリアプレット制御用 API
//! @{

/**
    @brief ライブラリアプレットとしての初期化を行ない、アプレットのメイン関数を呼び出します。

    @param[in] libraryAppletMain アプレットのメイン関数エントリ

    @details
     ライブラリアプレットとしての初期化を行い、その後 libraryAppletMain で指定された関数をアプレットのメイン関数として呼び出します。
     libraryAppletMain は LibraryAppletSelfInfo を引数にとります。
     この引数から以下の情報を取得可能です。

     - appletId: どの appletId のライブラリアプレットとして起動されたか
     - libraryAppletMode: どのライブラリアプレットモードで起動されたか

     本関数の呼び出し中、ライブラリアプレットとして実行中であることになります。

     本関数は nnMain() の中から呼び出してください。
     また、本関数を呼ぶまで、ae および applet ライブラリで提供している関数を呼びだすこと基本的にできません。
     ただし SetLibraryAppletStartHook() は本関数の呼び出しに先んじて呼ぶことが可能です。

     libraryAppletMain では、以下のような動作をすることを想定されています。

     - info.libraryAppletMode で指定されたモードによるチェック、および、必要に応じて分岐
     - TryPopFromInChannel() で入力データを取得
     - 実処理
     - PushToOutChannel() で出力データを設定
     - libraryAppletMain からの return または ExitLibraryApplet() の呼び出し

     info.isUnwound == false のときには、現 LA プログラムは通常起動されています。
     この場合 info.previousLibraryAppletHandle は無効です。

     info.isUnwound == true のときには、現 LA プログラムは、いわゆるスクラッチパッド起動により再起動したものです。
     info.previousLibraryAppletHandle には、直前に呼ばれていたライブラリアプレットのハンドルが格納されいます。
     この際のシーケンスは SetLibraryAppletStartHook() を参照してください。
*/
void InvokeLibraryAppletMain(void (*libraryAppletMain)(const LibraryAppletSelfInfo& info)) NN_NOEXCEPT;

/**
    @brief ライブラリアプレットをその場で終了します。

    @details
     本関数は InvokeLibraryAppletMain() で指定した関数の途中でライブラリアプレットの処理を正常終了します。
     本関数を呼ぶと InvokeLibraryAppletMain() で指定した関数を抜けた場合と同様の動作となります。
     本関数は処理を返しません。
*/
NN_NORETURN void ExitLibraryApplet() NN_NOEXCEPT;

/**
    @brief 呼び出し元のメインアプレット判別情報を取得します。

    @return 呼び出し元のメインアプレット判別情報を返します。

    @pre    自身がライブラリアプレットである
*/
applet::AppletIdentityInfo GetMainAppletIdentityInfo() NN_NOEXCEPT;

/**
    @brief  呼び出し元のアプレット判別情報を取得します。

    @return 呼び出し元のアプレット判別情報を返します。

    @pre    自身がライブラリアプレットである

    @details
     自ライブラリアプレットを呼び出したアプリケーションやアプレットの
     判別情報を applet::AppletIdentityInfo 構造体で取得します。
*/
applet::AppletIdentityInfo GetCallerAppletIdentityInfo() NN_NOEXCEPT;

/**
    @brief  自アプレット終了時の戻り先アプレット判別情報を取得します。

    @return 自アプレット終了時の戻り先アプレット判別情報を返します。

    @pre    自身がライブラリアプレットである

    @details
     自ライブラリアプレット終了時の戻り先となるアプリケーションやアプレットの
     判別情報を applet::AppletIdentityInfo 構造体で取得します。

     通常、この情報は nn::ae::GetCallerAppletIdentityInfo() で得られるものと
     同等であることが多いですが、 nn::applet::JumpLibraryApplet() 等で
     呼出元アプレットが Unwinding 起動せずに、その呼出元にまで直で戻るような
     ケースにおいて、呼出元とは異なる戻り先を返します。

*/
applet::AppletIdentityInfo GetNextReturnDestinationAppletIdentityInfo() NN_NOEXCEPT;


/**
    @brief      呼び出し元のメインアプレットのアプリケーション管理データを取得します。

    @param[out] pOut 呼び出し元のメインアプレットのアプリケーション管理データ。

    @retresult
    @endretresult

    @pre        - 自身がライブラリアプレットである
                - 呼び出し元がアプリケーションである

    @details
     成功した場合、outValue が指すポインタに取得したデータが格納されます。
*/
Result GetMainAppletApplicationControlProperty(ns::ApplicationControlProperty* pOut) NN_NOEXCEPT;

/**
    @brief  呼出元アプリケーションが要望する最も優先度の高い言語を取得します。

    @param[out] pOutLanguageCode    取得した言語コードの格納先アドレス

    @retresult
     @handleresult{nn::ae::ResultMainAppletNotApplication}
    @endretresult

    @details
     呼出元のメインアプレットがアプリケーションの場合に、
     そのアプリケーションが対応している言語のうち、本体設定で設定される言語と
     照らし合わせて最も適切な言語を取得し、pOutLanguageCode へ格納します。

     アプリケーションの言語設定に関しては、
     nn::oe::GetDesiredLanguage() も合わせて参考にして下さい。

     なお、メインアプレットがアプリケーションではない場合、
     返値として nn::ae::ResultMainAppletNotApplication() を返します。

*/
Result GetMainAppletApplicationDesiredLanguage(settings::LanguageCode* pOutLanguageCode) NN_NOEXCEPT;

/**
    @brief      呼び出し元のメインアプレットのストレージを取得します。

    @param[out] pOut 呼び出し元のメインアプレットのストレージ。

    @return     取得に成功した場合は true。失敗した場合は false。

    @pre        - 自身がライブラリアプレットである
                - 呼び出し元がアプリケーションである
*/
bool GetMainAppletStorageId(ncm::StorageId* pOut) NN_NOEXCEPT;

/**
    @brief アプリケーション用の CPU コアが使用可能かどうかを取得します。

    @return アプリケーション用の CPU コアが使用可能なら true、そうでないなら false を返します。

    @pre 自身がライブラリアプレットである

    @details
     アプリケーション用の CPU コアが使用可能かどうかを bool 値で返します。

*/
bool CanUseApplicationCore() NN_NOEXCEPT;


/**
    @brief   指定したアプリケーションの起動をシステムに要求します。

    @param[in] applicationId  起動したいアプリケーションの ApplicationId
    @param[in] buffer         アプリの起動パラメータを格納したバッファ
    @param[in] bufferSize     アプリの起動パラメータを格納したバッファのサイズ

    @pre 自身がライブラリアプレットである

    @details
     applicationId で指定されたアプリケーションプログラムを起動するように
     システムに要求を出します。本 API は要求を出した後にすぐにリターンします。

     buffer と bufferSize には、起動したアプリケーションに渡される
     起動パラメータを指定します。起動したアプリケーションは、
     nn::oe::TryPopLaunchParameter() にてこの起動パラメータを取得できます。

     プログラムの起動要求はシステムアプレットに通知されて処理されます。
     システムアプレットはこの通知を受けると、要求元のライブラリアプレットや
     起動中のアプリケーションなどを全て終了させた後に、
     applicationId で指定されたプログラムを起動します。

     そのため、呼出元であるライブラリアプレットは、本 API 発行後は、
     自身に対する終了要求通知に対して、正しく終了できるようにしておいて下さい。

*/
void RequestToLaunchApplication(ncm::ApplicationId applicationId, const void* buffer, size_t bufferSize) NN_NOEXCEPT;


//! @}

//! @name ライブラリアプレット入出力用 API
//! @{

/**
    @brief 入力チャンネルからストレージの pop を試みます。

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre 自身がライブラリアプレットである
    @pre *pOut に書き込める

    @post true が返ったとき、*pOut は有効で解放されていないストレージを表すハンドルである

    @details
     自身の入力チャンネルから pop を試み、
     成功すれば *pOut にストレージのハンドルを書き込み true を返します。
     pop できるデータがなかった場合には、false を返します。

     pop に成功したストレージの所有権と管理責任は、本関数の呼び出し元に移動します。
     使用し終わったら ReleaseStorage() に渡して解放する必要があります。
     また、必要に応じて、解放せずに別の Push 系関数に渡すことも可能です。
*/
bool TryPopFromInChannel(StorageHandle* pOut) NN_NOEXCEPT;

/**
    @brief インタラクティブ入力チャンネルからストレージの pop を試みます。

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre 自身がライブラリアプレットである
    @pre *pOut に書き込める

    @post true が返ったとき、*pOut は有効で解放されていないストレージを表すハンドルである

    @details
     自身ののインタラクティブ入力チャンネルから pop を試み、
     成功すれば *pOut にストレージのハンドルを書き込み true を返します。
     pop できるデータがなかった場合には、false を返します。

     pop に成功したストレージの所有権と管理責任は、本関数の呼び出し元に移動します。
     使用し終わったら ReleaseStorage() に渡して解放する必要があります。
     また、必要に応じて、解放せずに別の Push 系関数に渡すことも可能です。
*/
bool TryPopFromInteractiveInChannel(StorageHandle* pOut) NN_NOEXCEPT;

/**
    @brief インタラクティブ入力チャンネルに pop できるデータがあることを待機するためのイベントを取得します。

    @return イベントへのポインタを返します。

    @pre 自身がライブラリアプレットである

    @details
     自身のインタラクティブ入力チャンネルに pop できるデータがあることを待機するためのイベントを取得します。
     ここで取得されたイベントは、インタラクティブ入力チャンネルにストレージがあるときにシグナルされます。
     このイベントは TryPopFromInteractiveInChannel() の呼び出しによってストレージが空になると、シグナルがクリアされます。
*/
os::SystemEventType* GetPopFromInteractiveInChannelEvent() NN_NOEXCEPT;

/**
    @brief 出力チャンネルにストレージを push します。

    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。

    @details
     自身の出力チャンネルに、
     storageHandle で指定されるストレージを push します。

     storageHandle で表されていたストレージの所有権は、
     本関数を呼んだライブラリアプレットから外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void PushToOutChannel(StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief インタラクティブ出力チャンネルにストレージを push します。

    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。

    @details
     自身のインタラクティブ出力チャンネルに、
     storageHandle で指定されるストレージを push します。

     storageHandle で表されていたストレージの所有権は、
     本関数を呼んだライブラリアプレットから外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void PushToInteractiveOutChannel(StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief 指定された転送ストレージをメモリ空間上にマップします。

    @param[out] pOutAddress マップされたストレージの先頭アドレスの格納先
    @param[out] pOutSize    マップされたストレージのサイズの格納先
    @param[in]  handle      転送ストレージを指すストレージハンドル

    @pre 自身がライブラリアプレットである

    @retresult
        @handleresult{nn::applet::ResultNotTransferStorage}
        @handleresult{nn::applet::ResultAlreadyMappedTransferStorage}
        @handleresult{nn::applet::ResultTooManyTransferStorageMapping}
    @endretresult

    @details
     指定された handle の転送ストレージを自身のメモリ空間上にマップします。
     マップに成功した場合、pOutAddress にマップしたアドレスを、
     pOutSize にストレージのサイズを格納します。

     マップした転送ストレージをアンマップする際には
     nn::ae::UnmapTransferStorage() を使用して下さい。

     なお、自アプレットを終了する際には必ずアンマップしなければなりません。
     ストレージをマップした状態で自アプレットが終了してしまった場合は、
     FATAL エラーが発生します。

*/
Result MapTransferStorage(void** pOutAddress, size_t* pOutSize, StorageHandle handle) NN_NOEXCEPT;

/**
    @brief 指定された転送ストレージをメモリ空間上からアンマップします。

    @param[in]  handle  転送ストレージを指すストレージハンドル

    @pre 自身がライブラリアプレットである

    @details
     指定された転送ストレージをメモリ空間上からアンマップします。

     自アプレットを終了する前には、必ず本 API でアンマップして下さい。
     ストレージをマップした状態で自アプレットが終了してしまった場合は、
     FATAL エラーが発生します。

*/
void UnmapTransferStorage(StorageHandle handle) NN_NOEXCEPT;

//! @}

/**
    @brief ライブラリアプレット起動時のフック関数の返り値型
*/
enum LibraryAppletStartHookResult
{
    LibraryAppletStartHookResult_Normal, //!< 通常の動作でライブラリアプレット起動します
    LibraryAppletStartHookResult_WindProgram, //!< 現プログラムをいったん終了しライブラリアプレットを起動します
};

/**
    @brief ライブラリアプレット起動時のフック関数の型
*/
typedef LibraryAppletStartHookResult (*LibraryAppletStartHook)(applet::AppletId appletId, applet::LibraryAppletMode libraryAppletMode, void* userArgument);

//! @name スクラッチパッド制御用 API
//! @{

/**
    @brief ライブラリアプレット起動時のフックを設定します。

    @details
     本関数で設定された関数 hook は applet::StartLibraryApplet() の中で、
     起動しようとしている appletId と libraryAppletMode を伴って呼ばれます。
     また userArgument 引数の値がそのまま渡されます。

     hook では、以下の処理を行うことが期待されています。

     指定されたライブラリアプレットの起動が、自プログラムを一旦終了せずに起動できる場合、
     hook() は何もせずに LibraryAppletStartHookResult_Normal を返します。

     指定されたライブラリアプレットの起動を行うために、自プログラムを一旦終了する必要がある場合(スクラッチパッドの場合)、
     必要なコンテキスト保存(後述)を行った後、LibraryAppletStartHookResult_WindProgram を返します。
     この場合、起動しようとしているライブラリアプレットが起動される前に、自アプレットのプログラムが一旦終了し、
     ライブラリアプレットを起動します。
     起動したライブラリアプレットが終了すると、再度呼び出し元のプログラムが再起動します。
     LibraryAppletSelfInfo:IsUnwound は true となり、
     この際 LibraryAppletSelfInfo の previousLibraryAppletHandle には処理が終わったライブラリアプレットのハンドルが格納されています。
     この previousLibraryAppletHandle に対しては、通常どおりに applet::StartLibraryApplet() を呼んだ場合と同様に、
     applet::JoinLibraryApplet() を呼んだ後に、applet::TryPopFromOutChannel() 等を使用して、
     処理後のデータを取得することができます。

     コンテキストは、各ライブラリアプレットごとに用意される ContextStack という領域に、ストレージの形で保存を行います。
     ライブラリアプレットの各種入出力 Channel や ContextStack は、プログラムが一旦終了している間も引き続き維持されます。

     ContextStack へのpush は PushToContextStack() を呼ぶことで行います。
     コンテキストの復帰を行う際は PopFromContextStack() を呼ぶことで行います。

     また UnpopToInChannel() を使用することで、一旦 pop した InChannel のデータを戻しておき、
     再起動時に再度読むことも可能です。

     これらのコンテキスト保存に使用するストレージには制限があり、以下のものに限られます。

     - チャンネルから pop したストレージ
     - applet::CreateStorage() で作成したストレージ

     例えば InChannel から pop したストレージをそのまま ContextStack に push することは可能です。
     逆に、その他のストレージ、特に

     - 自プログラムで CreateLargeStorage() で作成したストレージ

     はコンテキスト保存に使用しないようにしてください(今後の実装変更により、エラーとなる予定です)。
*/
void SetLibraryAppletStartHook(LibraryAppletStartHook hook) NN_NOEXCEPT;

/**
    @brief コンテキストスタックにストレージを push します。

    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。
     自身のコンテキストスタックに、
     storageHandle で指定されるストレージを push します。

     storageHandle で表されていたストレージの所有権は、
     本関数を呼んだライブラリアプレットから外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void PushToContextStack(StorageHandle storageHandle) NN_NOEXCEPT;

/**
    @brief コンテキストスタックからストレージの pop を試みます。

    @param[out] pOut 作成されたストレージを表すストレージハンドルを格納するバッファポインタを指定します。

    @return pop した場合には true を返し、pop するデータがなかった場合には false を返します。

    @pre 自身がライブラリアプレットである
    @pre *pOut に書き込める

    @post true が返ったとき、*pOut は有効で解放されていないストレージを表すハンドルである

    @details
     自身コンテキストスタックから pop を試み、
     成功すれば *pOut にストレージのハンドルを書き込み true を返します。
     pop できるデータがなかった場合には、false を返します。

     pop に成功したストレージの所有権と管理責任は、本関数の呼び出し元に移動します。
     使用し終わったら ReleaseStorage() に渡して解放する必要があります。
     また、必要に応じて、解放せずに別の Push 系関数に渡すことも可能です。
*/
bool TryPopFromContextStack(StorageHandle* pOut) NN_NOEXCEPT;

/**
    @brief 入力チャンネルにストレージを unpop します。

    @param[in] storageHandle 対象のストレージのハンドルを指定します。

    @pre storageHandle が有効で、まだ解放されていないストレージのハンドルである。

    @post storageHandle は無効になる。

    @details
     自身の入力チャンネルに、
     storageHandle で指定されるストレージを unpop します。
     次に PopFromInChannel() を呼んだときに、unpop したものと同じストレージを得ることができます。(ハンドルは同一になるとは限りません)

     storageHandle で表されていたストレージの所有権は、
     本関数を呼んだライブラリアプレットから外れ、アクセスできなくなります。
     storageHandle に対して ReleaseStorage() を呼ぶ必要もありません。
*/
void UnpopToInChannel(StorageHandle storageHandle) NN_NOEXCEPT;

//! @}

//! @name 終了時ハンドリング
//! @{

/**
    @brief 終了時ハンドリングを行うことを宣言します。

    @details
     本回数を呼んだ回数が DisableHandlingForExitRequest() を呼んだ回数より多い場合、
     終了時ハンドリングが有効になります。

     終了時ハンドリングが有効なときに、
     RequestExitLibraryApplet 系関数が呼ばれたとき、その対象であった場合 Message_Exit が通知されます。
     この通知に対して以下のハンドリングを行ってください。

     - ExitLibraryApplet() を呼ぶことで正常終了することができ、呼び出し元に LibraryAppletExitReason_Normal が返ります
     - 必要な回数 DisableHandlingForExitRequest() を呼ぶことで、キャンセル終了とすることができ、呼び出し元に LibraryAppletExitReason_Canceled が返ります
     - いずれでもない場合に、タイムアウトして強制終了されたり、途中でアボートしたような場合には、呼び出し元に LibraryAppletExitReason_Abnormal が返ります

     終了時ハンドリングが有効でないとき、すなわち、本関数を呼んだ回数が DisableHandlingForExitRequest() より少ないとき、
     即座にアプレットが強制的に終了されます。
     この場合、呼び出し元には LibraryAppletExitReason_Canceled が返ります。
*/
void EnableHandlingForExitRequest() NN_NOEXCEPT;

/**
    @brief 終了時ハンドリングをやめることを宣言します。

    @pre (EnableHandlingForExitRequest() を呼んだ回数) > (これまでに DisableHandlingForExitRequest() を呼んだ回数)

    @see EnableHandlingForExitRequest()
*/
void DisableHandlingForExitRequest() NN_NOEXCEPT;

//! @}


//! @name コントローラ F/W 更新区間の制御
//! @{

/**
    @brief  コントローラ F/W 更新区間を開始します。

    @pre
     - 自身がライブラリアプレットである
     - nn::ae::IsInControllerFirmwareUpdateSection() == false

    @details
     コントローラ F/W 更新区間を開始します。
     この区間は EndControllerFirmwareUpdateSection() を発行すると終了します。

     本 API によりコントローラ F/W 更新区間が開始されると、SA および OA に
     nn::ae::Message_ControllerFirmwareUpdateSectionChanged が通知されます。
     同様にこの区間を終了した時にも、このメッセージが通知されます。

     本 API と EndControllerFirmwareUpdateSection() は、コントローラ F/W
     更新区間の開始や終了をシステムに通知する以外の効果はありません。
     実際に HOME ボタンや POWER ボタンを無効化する実装は、
     SA や OA にて上記通知メッセージをハンドリングして行うようにして下さい。

*/
void BeginControllerFirmwareUpdateSection() NN_NOEXCEPT;

/**
    @brief  コントローラ F/W 更新区間を終了します。

    @pre
     - 自身がライブラリアプレットである
     - nn::ae::IsInControllerFirmwareUpdateSection() == true

    @see BeginControllerFirmwareUpdateSection()
*/
void EndControllerFirmwareUpdateSection() NN_NOEXCEPT;

//! @}


//! @name (開発用) ライブラリアプレット開発支援 API
//! @{

/**
    @brief (開発用) ライブラリアプレット単体起動時のライブラリアプレット作成関数を設定します。

    @param[in] createSelfLibraryApplet ライブラリアプレットを作成する関数を指定します。

    @pre InvokeLibraryAppletMain() の呼び出し直前

    @details
     ライブラリアプレットの単体起動時には、起動されたプログラムがどのようなライブラリアプレットであるかを認識できません。
     このため、システムに対し、どのようにして作成されたライブラリアプレットとして単体起動されたのかを認識させる必要があります。
     本関数は、このような場合に、システムがどう認識すればよいかを指定するためのものです。

     本関数を InvokeLibraryAppletMain() に先行して呼びだしていた場合、
     InvokeLibraryAppletMain() 内でライブラリアプレットメイン関数が呼ばれる前に以下のような処理が行われます。

     - 本関数を呼びだしたプログラムが、ライブラリアプレットとしてシステムに認識されているかを確認する
     - 認識されていれば通常どおり進む。以下は、認識されていなかった場合
     - createSelfLibraryApplet() を呼んでライブラリアプレットハンドルを取得する
     - 取得したものに対応するライブラリアプレットが、本関数を呼びだしたプログラムに対応するものとして、システムが認識する

     このため createSelfLibraryApplet 内では、以下の処理をすることが想定されています。

     - applet::CreateLibraryApplet() を呼んでライブラリアプレットを作成し、ハンドルを取得する
     - 必要に応じて、このハンドルに対し applet::PushToInChannel() などを呼んで、入力データなどを設定する
     - このハンドルを返す
     - ※ applet::StartLibraryApplet() は呼ばない

     createSelfLibraryApplet 内は、特殊な環境で呼び出されるため、
     nn::ae の関数を呼ぶことはできません。
     また、nn::applet の関数も使用が制限されます。
     nn::applet の関数で呼ぶことができるものは以下のものです。

     - ライブラリアプレットの作成 (applet::CreateLibraryApplet())
     - ストレージ操作関数 (applet::CreateStorage() など)
     - 入力チャンネルアクセス (applet::PushToInChannel())
*/
void SetCreateSelfLibraryAppletForDevelop(applet::LibraryAppletHandle (*createSelfLibraryApplet)()) NN_NOEXCEPT;

//! @}

//! @name 暫定 API
//! @{

/**
    @brief (暫定 API) 自ライブラリアプレットの IndirectLayer の ProducerHandle を取得します。

    @return 自ライブラリアプレットの IndirectLayer の ProducerHandle を返します。

    @details
     先行開発用の暫定 API です。

     LibraryAppletMode_PartialForegroundWithIndirectDisplay で起動されたライブラリアプレットからのみ呼ぶことができます。
     本関数の返り値を vi::OpenIndirectLayer() に渡すことを想定しています。
*/
vi::IndirectProducerHandleType GetIndirectLayerProducerHandle() NN_NOEXCEPT;

//! @}

//! @name エラーレポート API
//! @{

/**
    @brief      エラー履歴の表示対象となるエラーレポートを保存します。

    @param[in]  errorCode エラーコード

    @details
     エラービューアを利用せずに独自の方法でエラーを表示する Cruiser 向けの機能です。
     エラーメッセージのシステムデータにメッセージが登録されている必要があります。
*/
void ReportVisibleError(err::ErrorCode errorCode) NN_NOEXCEPT;

/**
    @brief      エラー履歴の表示対象となるエラーレポートを保存します。

    @param[in]  errorCode エラーコード
    @param[in]  errorContext エラーコンテキスト

    @details
     エラービューアを利用せずに独自の方法でエラーを表示する Cruiser 向けの機能です。
     エラーメッセージのシステムデータにメッセージが登録されている必要があります。
*/
void ReportVisibleError(err::ErrorCode errorCode, const err::ErrorContext& errorContext) NN_NOEXCEPT;

//! @}


//! @name   無線通信のプライオリティに関する API
//! @{

/**
 * @brief   無線通信のプライオリティを設定します。
 *
 * @param[in]  wirelessPrioriryMode  無線通信のプライオリティを表す列挙型
 *
 * @pre
 *  - 自身がライブラリアプレットである
 *
 * @details
 *  互いに影響を与え合う 2.4GHz 帯無線通信において、
 *  どの通信を優先するかを WirelessPriorityMode 型で設定します。
 *  本 API の効果は自アプレットが Foreground の場合にのみ効果があります。
 *
 *  デフォルトでもある WirelessPriorityMode_Default が指定された場合は、
 *  呼出元アプレット（アプリケーション含む）の設定値に従うことを意味します。
 *
 */
void SetWirelessPriorityModeForLibraryApplet(WirelessPriorityMode wirelessPrioriryMode) NN_NOEXCEPT;


//! @}

}} // namespace nn::ae
