﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   各アプレットの状態遷移の制御に関する API の宣言
 *
 * @details
 *  本ヘッダでは、各アプレットが状態遷移の制御に関する API を定義しています。
 *
 *  本ヘッダに記載の機能は、各種アプレット開発者のみが利用できます。
 *
 *  各アプレットは Foreground 状態と Background 状態のいずれかの状態を取り、
 *  起動直後は Background 状態です。
 *
 *  Foreground 状態にあるアプレットは、GPU による描画や、サウンド出力、
 *  パッド入力の取得などを行なうことができます。
 *
 *  また、各アプレットはアプレットマネージャから、Foreground 状態や
 *  Background 状態への遷移を示す通知メッセージを受け取ります。
 *
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/ae/ae_Types.h>
#include <nn/applet/applet_Types.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/diag/diag_AbortObserver.h>

namespace nn { namespace ae {

/**
 * @brief   キャプチャバッファのインデックス値を示す列挙型です。
 *
 */
enum CaptureBufferIndex
{
    CaptureBufferIndex_LastApplication = applet::CaptureBufferIndex_LastApplication,    //!< LastApplication キャプチャバッファを示すインデックス値です。
    CaptureBufferIndex_LastForeground  = applet::CaptureBufferIndex_LastForeground,     //!< LastForeground キャプチャバッファを示すインデックス値です。
    CaptureBufferIndex_CallerApplet    = applet::CaptureBufferIndex_CallerApplet,       //!< CallerApplet キャプチャバッファを示すインデックス値です。
};


//! @name 各種アプレット開発用 API
//! @{


//--------------------------------------------------------------------------
/**
 * @brief   通知メッセージ受信可能を知らせる os::SystemEventType オブジェクトを初期化します。
 *
 * @param[in]   event   os::SystemEventType オブジェクトへのポインタ
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - event が初期化されていない状態である
 *
 * @details
 *  アプレットマネージャからの通知メッセージが受信可能な場合にシグナル化される
 *  nn::os::SystemEventType オブジェクトとして event を初期化します。
 *
 *  event には未初期化の nn::os::SystemEventType オブジェクトを渡して下さい。
 *  この event は、アプレットマネージャから通知メッセージを受け取ると
 *  シグナル化されます。アプレット開発者は、nn::os::WaitSystemEvent() や
 *  nn::os::WaitAny() などを使ってこれらのシグナル化を待ち受け、
 *  GetNotificationMessage() を使ってメッセージを受け取り、
 *  メッセージの内容に沿った処理を行なって下さい。
 *
 *  なお、event は nn::os::EventClearMode_ManualClear で初期化されます。
 *  全ての通知メッセージを受け取ると event は自動的にクリアされるため、
 *  アプレット開発者が event を明示的にクリアする必要はありません。
 *
 */
void InitializeNotificationMessageEvent(os::SystemEventType* event) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   自アプレットへの通知メッセージを取得します。
 *
 * @return  通知メッセージを示す Message 型の列挙子が返ります。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *
 * @details
 *  自アプレットへ届いている通知メッセージを取得します。
 *
 *  Initialize() で渡した SystemEventType オブジェクトがシグナル化されると、
 *  アプレットマネージャから自アプレットに対して何かしらの通知メッセージが
 *  送られて来ています。本関数はその通知メッセージを取得します。
 *
 *  全ての通知メッセージを取得し終わると、Initialize() 時に渡した
 *  SystemEventType オブジェクトのシグナルは自動的にクリアされます。
 *
 *  自アプレットに通知メッセージが届いていない場合、
 *  本関数は nn::ae::Message_None を返します。
 *
 *  各通知メッセージの意味は以下の通りです。
 *  以下に各メッセージを受取った後に処理すべき内容を簡単に記載しています。
 *
 *  - nn::ae::Message_ChangeIntoForeground
 *    - Foreground 状態に遷移したことを表します。
 *  - nn::ae::Message_ChangeIntoBackground
 *    - Background 状態に遷移したことを表します。
 *  - nn::ae::Message_Exit
 *    - 自アプレットへの終了要求を表します。
 *    - 終了に必要な処理を行なった後、nnMain() からリターンすることで自アプレットを終了できます。
 *  - nn::ae::Message_None
 *    - 特にすべきことはありません。
 *    - 次の Message を待機して下さい。
 *  - 上記以外の Message を受け取った場合
 *    - Message の内容に沿って然るべき処理を行なって下さい。
 *    - 詳細は別途開発マニュアル等を参照して下さい。
 *
 */
Message GetNotificationMessage() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief この API は従来との互換性を維持するために残してあります。
 *
 * @details
 *  本 API は何も行いません。従来との互換性のために残してあるものです。
 *  新規に開発されるアプレットでは、本 API を使用する必要はありません。
 *
 */
void AcquireForegroundRights() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief この API は従来との互換性を維持するために残してあります。
 *
 * @details
 *  本 API は何も行いません。従来との互換性のために残してあるものです。
 *  新規に開発されるアプレットでは、本 API を使用する必要はありません。
 *
 */
void ReleaseForegroundRights() NN_NOEXCEPT;


//! @}

//-----------------------------------------------------------------------------

//! @name 各種アプレット開発用 API ユーティリティ
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   自アプレットへ通知メッセージが届くのを待機し、そのメッセージを取得します。
 *
 * @param[in] event     nn::os::SystemEventType オブジェクトへのポインタ
 *
 * @return  通知メッセージを示す Message 型の列挙子が返ります。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *
 * @details
 *  自アプレットへ通知メッセージが届くのを待機し、届いたメッセージを返します。
 *
 *  引数の event には、nn::ae::InitializeNotificationMessageEvent() で
 *  初期化した SystemEventType オブジェクトを渡して下さい。
 *  本関数は、この event がシグナル化されるのを待機し、
 *  シグナル化を検知した後の振舞いは、GetNotificationMessage() と同様です。
 *  詳細はそちらを参照して下さい。
 *  ただし、 Message_None が返値として返されることはありません。
 *
 *  本関数は event の待機と GetNotificationMessage() を組み合わせた
 *  ユーティリティ関数です。特に、各種アプレットを開発する上で、
 *  実装を容易にするために用意されています。
 *
 *  以下にメッセージ通知を受け取るメインループの実装サンプルを示します。
 *
 *  @code
 *  #include <nn/nn_Abort.h>
 *  #include <nn/os.h>
 *  #include <nn/os/os_SystemEvent.h>
 *  #include <nn/ae.h>
 *
 *  void SystemAppletMenuMain(nn::ae::AppletParameter* param)
 *  {
 *      nn::os::SystemEventType event;
 *      nn::ae::InitializeNotificationMessageEvent(&event);
 *
 *      // メインループ
 *      for (;;)
 *      {
 *          auto message = nn::ae::WaitForNotificationMessage(&event);
 *          switch (message)
 *          {
 *              case nn::ae::Message_ChangeIntoForeground:
 *                      // Foreground 状態に遷移したときの処理を行なう
 *                      break;
 *
 *              case nn::ae::Message_ChangeIntoBackground:
 *                      // Background 状態に遷移したときの処理を行なう
 *                      break;
 *
 *              case nn::ae::Message_*:
 *                      // 各メッセージに対応した実装が必要
 *                      break;
 *
 *              case nn::ae::Message_Exit:
 *                      // 自アプレットの終了処理を行なう
 *                      return;
 *
 *              default:
 *                      // 未知のメッセージは無視する
 *                      break;
 *          }
 *      }
 *  }
 *
 *  extern "C" void nnMain()
 *  {
 *      nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain);
 *  }
 *
 *  @endcode
 *
 */
Message WaitForNotificationMessage(nn::os::SystemEventType* event) NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

//! @name 出画画像の操作に関する API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像を取得します。
 *
 * @param[out]  pOutIsScreenShotEnabled アプリケーションのキャプチャ撮影許可状態の格納先
 * @param[out]  pOutBuffer  画像イメージを格納するバッファへのポインタ
 * @param[in]   dataSize    画像イメージを格納するバッファのサイズ
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - dataSize == CaptureBufferSize
 *
 * @details
 *  アプリケーションが FG 状態でなくなった瞬間の最終出画画像を取得します。
 *  アプリケーションがスクリーンショット撮影を禁止していた場合でも取得可能で、
 *  実際の撮影許可状態は pOutScreenShotEnabled に格納されます。
 *
 *  一方、以下のいずれかの API でキャプチャ画像をマップしている場合など、
 *  システムが何等かの理由でキャプチャ画像を参照できない場合には、
 *  本 API は nn::ae::ResultCaptureBufferBusy を返し、画像は取得されません。
 *
 *  - nn::ae::MapLastApplicationCaptureBuffer()
 *  - nn::ae::MapLastForegroundCaptureBuffer()
 *  - nn::ae::MapCallerAppletCaptureBuffer()
 *
 *  取得した画像データの詳細は以下の通りです。
 *
 *  - 解像度は 1280 x 720
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm 相当
 *
 */
Result GetLastApplicationCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   直前の Foreground アプレットの最終出画画像を取得します。
 *
 * @param[out]   pOutIsScreenShotEnabled 対象アプレットのキャプチャ撮影許可状態の格納先
 * @param[out]   pOutBuffer   画面バッファを格納するバッファへのポインタ
 * @param[in]    dataSize     画面バッファを格納するバッファのサイズ
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - dataSize == CaptureBufferSize
 *
 * @details
 *  直前に Foreground 状態だったアプレットの最終出画画像を取得します。
 *  対象アプレットがスクリーンショット撮影を禁止していた場合でも取得可能で、
 *  実際の撮影許可状態は pOutScreenShotEnabled に格納されます。
 *
 *  一方、以下のいずれかの API でキャプチャ画像をマップしている場合など、
 *  システムが何等かの理由でキャプチャ画像を参照できない場合には、
 *  本 API は nn::ae::ResultCaptureBufferBusy を返し、画像は取得されません。
 *
 *  - nn::ae::MapLastApplicationCaptureBuffer()
 *  - nn::ae::MapLastForegroundCaptureBuffer()
 *  - nn::ae::MapCallerAppletCaptureBuffer()
 *
 *  取得した画像データの詳細は以下の通りです。
 *
 *  - 解像度は 1280 x 720
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm 相当
 *
 */
Result GetLastForegroundCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   現在の出画画像を直前の Foreground アプレットの最終出画画像としてシステムに保持させます。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *
 * @details
 *  現在の出画画像を直前の Foreground アプレットの最終出画画像としてシステムに保持させます。
 *
 *  以下のいずれかの API で、いずれかのキャプチャ画像をマップ中の場合は
 *  本 API は何も行ないません。
 *
 *  - nn::ae::MapLastApplicationCaptureBuffer()
 *  - nn::ae::MapLastForegroundCaptureBuffer()
 *  - nn::ae::MapCallerAppletCaptureBuffer()
 *
 */
void HoldCurrentCaptureBufferAsLastForeground() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレット呼出元画像を取得します。
 *
 * @param[out]   pOutIsScreenShotEnabled 対象アプレットのキャプチャ撮影許可状態の格納先
 * @param[out]   pOutBuffer   画面バッファを格納するバッファへのポインタ
 * @param[in]    dataSize     画面バッファを格納するバッファのサイズ
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 *   @handleresult{nn::ae::ResultScreenShotDisabled}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - dataSize == CaptureBufferSize
 *
 * @details
 *  nn::applet::TakeScreenShotOfCallerApplet() によってキャプチャされた
 *  画像を取得します。通常これはライブラリアプレット呼出時に、
 *  呼出元の画像を呼出先に渡す目的で使用することを想定しています。
 *
 *  対象アプリケーションもしくは対象アプレットがスクリーンショット撮影を
 *  禁止していた場合でもキャプチャ画像は取得可能で、
 *  実際の撮影許可状態は pOutScreenShotEnabled に格納されます。
 *
 *  一方、以下のいずれかの API でキャプチャ画像をマップしている場合など、
 *  システムが何等かの理由でキャプチャ画像を参照できない場合には、
 *  本 API は nn::ae::ResultCaptureBufferBusy を返し、画像は取得されません。
 *
 *  - nn::ae::MapLastApplicationCaptureBuffer()
 *  - nn::ae::MapLastForegroundCaptureBuffer()
 *  - nn::ae::MapCallerAppletCaptureBuffer()
 *
 *  取得した画像データの詳細は以下の通りです。
 *
 *  - 解像度は 1280 x 720
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm 相当
 *
 */
Result GetCallerAppletCaptureBuffer(bool* pOutIsScreenShotEnabled, void* pOutBuffer, size_t dataSize) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   キャプチャバッファ間で画像データのコピーを行ないます。
 *
 * @param[in]  dstIndex 転送先のキャプチャバッファを示すインデックス値
 * @param[in]  srcIndex 転送元のキャプチャバッファを示すインデックス値
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  システムが持つキャプチャバッファ間での画像データのコピーを行ないます。
 *  dstIndex および srcIndex には、 nn::ae::CaptureBufferIndex の列挙子を
 *  指定して下さい。
 *
 *  いずれかのキャプチャバッファがシステムによって一時的に使用中の場合には、
 *  nn::ae::ResultCaptureBufferBusy が返ります。このような場合には、
 *  間を空けながら一定時間リトライを行なうなどの対策を行なって下さい。
 *
 */
Result CopyBetweenCaptureBuffers(CaptureBufferIndex dstIndex, CaptureBufferIndex srcIndex) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像のイメージデータをマップします。
 *
 * @param[out]  pOutIsScreenShotEnabled アプリケーションのキャプチャ撮影許可状態の格納先
 * @param[out]  pOutAddress 画像イメージをマップしたアドレスの格納先
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - いずれのアプレットも同画像イメージをマップしていない
 *
 * @details
 *  アプリケーションが FG 状態でなくなった瞬間の最終出画画像のイメージデータを
 *  メモリ空間上にマップし、そのマップアドレスを pOutAddress に格納します。
 *  本 API は GetLastApplicationCaptureBuffer() で取得できる画像と
 *  同じ画像データをメモリマップ方式で参照できるようにするためのものです。
 *
 *  本 API ではアプリケーションのスクリーンショット撮影が禁止されている場合でも
 *  対象のキャプチャ画像イメージをマップして参照することができます。
 *  このとき、アプリケーションの撮影許可状態を pOutScreenShotEnabled に返します。
 *
 *  ただし、いずれかのアプレットやシステムによってイメージデータ領域が
 *  使用されていた場合には nn::ae::ResultCaptureBufferBusy が返されます。
 *  この場合には、一定時間（1～10msec 程度）経ってから再度本 API でマップを
 *  試してください。
 *
 *  なお、本 API でマップしている間は、システム側でも当該キャプチャメモリを
 *  使用できなくなるため、必要な処理を終えたら速やかにアンマップして下さい。
 *
 *  本 API でマップを行なうと、その間、以下の制限がかかります。
 *
 *  - nn::ae::GetLastApplicationCaptureBuffer() は false を返します。
 *  - nn::ae::GetLastForegroundCaptureBuffer() は false を返します。
 *  - nn::ae::GetCallerAppletCaptureBuffer() は false を返します。
 *  - nn::ae::HoldCurrentCaptureBufferAsLastForeground() は何も行ないません。
 *  - nn::applet::TakeScreenShotOfCallerApplet() は何も行ないません。
 *
 *  上記の制限は UnmapLastApplicationCaptureBuffer() によって解除されます。
 *
 *  マップされた画像データのメモリ内容は以下の通りです。
 *
 *  - マップされたサイズは nn::ae::CaptureBufferSize
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm
 *
 */
Result MapLastApplicationCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像のイメージデータをアンマップします。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *
 * @details
 *  MapLastApplicationCaptureBuffer() でマップしていたイメージデータをアンマップします。
 *  MapLastApplicationCaptureBuffer() によって発生していた制限は解除されます。
 *
 */
void UnmapLastApplicationCaptureBuffer() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   直前の Foregounrd の最終出画画像のイメージデータをマップします。
 *
 * @param[out]  pOutIsScreenShotEnabled 対象アプレットのキャプチャ撮影許可状態の格納先
 * @param[out]  pOutAddress 画像イメージをマップしたアドレスの格納先
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - いずれのアプレットも同画像イメージをマップしていない
 *
 * @details
 *  直前に Foreground だったアプレットの最終出画画像のイメージデータを
 *  メモリ空間上にマップし、そのマップアドレスを pOutAddress に格納します。
 *  本 API は GetLastForegroundCaptureBuffer() で取得できる画像と
 *  同じ画像データをメモリマップ方式で参照できるようにするためのものです。
 *
 *  本 API では対象アプレットのスクリーンショット撮影が禁止されている場合でも
 *  対象のキャプチャ画像イメージをマップして参照することができます。
 *  このとき、対象アプレットの撮影許可状態を pOutScreenShotEnabled に返します。
 *
 *  ただし、いずれかのアプレットやシステムによってイメージデータ領域が
 *  使用されていた場合には nn::ae::ResultCaptureBufferBusy が返されます。
 *  この場合には、一定時間（1～10msec 程度）経ってから再度本 API でマップを
 *  試してください。
 *
 *  なお、本 API でマップしている間は、システム側でも当該キャプチャメモリを
 *  使用できなくなるため、必要な処理を終えたら速やかにアンマップして下さい。
 *
 *  本 API でマップを行なうと、その間、以下の制限がかかります。
 *
 *  - nn::ae::GetLastApplicationCaptureBuffer() は false を返します。
 *  - nn::ae::GetLastForegroundCaptureBuffer() は false を返します。
 *  - nn::ae::GetCallerAppletCaptureBuffer() は false を返します。
 *  - nn::ae::HoldCurrentCaptureBufferAsLastForeground() は何も行ないません。
 *  - nn::applet::TakeScreenShotOfCallerApplet() は何も行ないません。
 *
 *  上記の制限は UnmapLastForegroundCaptureBuffer() によって解除されます。
 *
 *  マップされた画像データのメモリ内容は以下の通りです。
 *
 *  - マップされたサイズは nn::ae::CaptureBufferSize
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm
 *
 */
Result MapLastForegroundCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像のイメージデータをアンマップします。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *
 * @details
 *  MapLastForegroundCaptureBuffer() でマップしていたイメージデータをアンマップします。
 *  MapLastForegroundCaptureBuffer() によって発生していた制限は解除されます。
 *
 */
void UnmapLastForegroundCaptureBuffer() NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレット呼出元画像のイメージデータをマップします。
 *
 * @param[out]  pOutIsScreenShotEnabled 対象アプレットのキャプチャ撮影許可状態の格納先
 * @param[out]  pOutAddress 画像イメージをマップしたアドレスの格納先
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *  - いずれのアプレットも同画像イメージをマップしていない
 *
 * @details
 *  ライブラリアプレット呼出元画像のイメージデータを
 *  メモリ空間上にマップし、そのマップアドレスを pOutAddress に格納します。
 *  本 API は GetCallerAppletCaptureBuffer() で取得できる画像と
 *  同じ画像データをメモリマップ方式で参照できるようにするためのものです。
 *
 *  本 API では呼出元アプレットのスクリーンショット撮影が禁止されている場合でも
 *  対象のキャプチャ画像イメージをマップして参照することができます。
 *  このとき、対象アプレットの撮影許可状態を pOutScreenShotEnabled に返します。
 *
 *  ただし、いずれかのアプレットやシステムによってイメージデータ領域が
 *  使用されていた場合には nn::ae::ResultCaptureBufferBusy が返されます。
 *  この場合には、一定時間（1～10msec 程度）経ってから再度本 API でマップを
 *  試してください。
 *
 *  なお、本 API でマップしている間は、システム側でも当該キャプチャメモリを
 *  使用できなくなるため、必要な処理を終えたら速やかにアンマップして下さい。
 *
 *  本 API でマップを行なうと、その間、以下の制限がかかります。
 *
 *  - nn::ae::GetLastApplicationCaptureBuffer() は false を返します。
 *  - nn::ae::GetLastForegroundCaptureBuffer() は false を返します。
 *  - nn::ae::GetCallerAppletCaptureBuffer() は false を返します。
 *  - nn::ae::HoldCurrentCaptureBufferAsLastForeground() は何も行ないません。
 *  - nn::applet::TakeScreenShotOfCallerApplet() は何も行ないません。
 *
 *  上記の制限は UnmapCallerAppletCaptureBuffer() によって解除されます。
 *
 *  マップされた画像データのメモリ内容は以下の通りです。
 *
 *  - マップされたサイズは nn::ae::CaptureBufferSize
 *  - イメージフォーマットは nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm
 *
 */
Result MapCallerAppletCaptureBuffer(bool* pOutIsScreenShotEnabled, void** pOutAddress) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレット呼出元画像のイメージデータをアンマップします。
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *    - オーバーレイアプレットとして実行中である
 *
 * @details
 *  MapCallerAppletCaptureBuffer() でマップしていたイメージデータをアンマップします。
 *  MapCallerAppletCaptureBuffer() によって発生していた制限は解除されます。
 *
 */
void UnmapCallerAppletCaptureBuffer() NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief キャプチャバッファをクリアする時のデフォルトカラー（黒）を示す定数です。
 *
 */
const Bit32 DefaultClearColorOfCaptureBuffer = 0xFF000000;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像キャプチャバッファを指定されたカラーで塗り潰します。
 *
 * @param[in]  isScreenShotEnabled  キャプチャ撮影許可状態
 * @param[in]  clearColor           キャプチャバッファを塗り潰すカラー値
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  アプリケーションの最終出画画像用のキャプチャバッファを、
 *  指定された clearColor で塗り潰し、
 *  この塗り潰した画像の撮影可否フラグを isScreenShotEnabled に設定します。
 *
 *  第 2 引数の clearColor を省略した場合は、塗り潰すカラー値として
 *  DefaultClearColorOfCaptureBuffer （黒）が適用されます。
 *
 */
Result ClearLastApplicationCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor = DefaultClearColorOfCaptureBuffer) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプリケーションの最終出画画像キャプチャバッファを指定されたカラーで塗り潰します。
 *
 * @param[in]  isScreenShotEnabled  キャプチャ撮影許可状態
 * @param[in]  clearColor           キャプチャバッファを塗り潰すカラー値
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  直前に FG 状態だったアプレットの最終出画画像用のキャプチャバッファを、
 *  指定された clearColor で塗り潰し、
 *  この塗り潰した画像の撮影可否フラグを isScreenShotEnabled に設定します。
 *
 *  第 2 引数の clearColor を省略した場合は、塗り潰すカラー値として
 *  DefaultClearColorOfCaptureBuffer （黒）が適用されます。
 *
 */
Result ClearLastForegroundCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor = DefaultClearColorOfCaptureBuffer) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   ライブラリアプレット呼出元画像キャプチャバッファを指定されたカラーで塗り潰します。
 *
 * @param[in]  isScreenShotEnabled  キャプチャ撮影許可状態
 * @param[in]  clearColor           キャプチャバッファを塗り潰すカラー値
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  ライブラリアプレット呼出元画像用のキャプチャバッファを、
 *  指定された clearColor で塗り潰し、
 *  この塗り潰した画像の撮影可否フラグを isScreenShotEnabled に設定します。
 *
 *  第 2 引数の clearColor を省略した場合は、塗り潰すカラー値として
 *  DefaultClearColorOfCaptureBuffer （黒）が適用されます。
 *
 */
Result ClearCallerAppletCaptureBuffer(bool isScreenShotEnabled, Bit32 clearColor = DefaultClearColorOfCaptureBuffer) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   アプレット間遷移時の出画画像バッファを指定されたカラーで塗り潰します。
 *
 * @param[in]  clearColor   キャプチャバッファを塗り潰すカラー値
 *
 * @retresult
 *   @handleresult{nn::ae::ResultCaptureBufferBusy}
 * @endretresult
 *
 * @details
 *  アプリや各アプレット間の遷移時に、それらの繋ぎ画像として使用される
 *  システムが持つ画像バッファを、指定された clearColor で塗り潰します。
 *
 *  引数の clearColor を省略した場合は、塗り潰すカラー値として
 *  DefaultClearColorOfCaptureBuffer （黒）が適用されます。
 *
 */
Result ClearAppletTransitionTextureBuffer(Bit32 clearColor = DefaultClearColorOfCaptureBuffer) NN_NOEXCEPT;


/**
    @brief アボート時にすべてのキャプチャバッファを開放する AbortObserver を取得します。

    @return アボート時にすべてのキャプチャバッファを開放する AbortObserver を返します。

    @details
     本関数を呼ぶことで、アボート時にすべてのキャプチャバッファを開放するような AbortObserver を取得できます。
     必要に応じて、本関数の返り値を diag::RegisterAbortObserver 関数に渡すことで、
     アボート時にすべてのキャプチャバッファを開放できるようになります。
*/
diag::AbortObserverHolder* GetAbortObserverHolderToUnmapCaptureBuffer() NN_NOEXCEPT;

//! @}

//-----------------------------------------------------------------------------

//! @name スクリーンショット機能に関する API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   ユーザによるスクリーンショット撮影の禁止許可状態を変更します。
 *
 * @param[in]   permission  撮影の禁止許可状態を示す列挙子
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *
 * @details
 *  自アプレットの出画画面のスクリーンショット撮影の禁止許可状態を
 *  permission に変更します。permission には以下のものが使用できます。
 *
 *  - nn::ae::ScreenShotPermission_Inherit （呼出元の撮影禁止許可状態を継承）
 *  - nn::ae::ScreenShotPermission_Permit  （撮影を許可）
 *  - nn::ae::ScreenShotPermission_Forbid  （撮影を禁止）
 *
 *  デフォルトでは nn::ae::ScreenShotPermission_Inherit となっています。
 *  また、システムアプレットにおける ScreenShotPermission_Inherit は、
 *  撮影許可を意味します。
 *
 */
void SetScreenShotPermission(ScreenShotPermission permission) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   スクリーンショット撮影ファイルに適用する ProgramId を設定します。
 *
 * @param[in]   info  撮影した画像ファイルに適用するアプレット情報
 *
 * @pre
 *  - 以下のいずれかの状態である
 *    - システムアプレットとして実行中である
 *    - ライブラリアプレットとして実行中である
 *
 * @details
 *   スクリーンショット撮影された画像ファイルに適用される
 *   ProgramId を設定します。自アプレット起動時のデフォルトは、
 *   自アプレットの ProgramId が設定されています。
 *
 *   ProgramId は applet::AppletIdentityInfo を使って指定します。
 *   そのため、info には nn::ae::GetMainAppletIdentityInfo() や
 *   nn::ae::GetCallerAppletIdentityInfo() の返値をそのまま指定できます。
 *
 *   info.appletId == applet::AppletId_Application の場合には、
 *   info.applicationId を ProgramId に変換して設定します。
 *   info.appletId != applet::AppletId_Application の場合には、
 *   info.applicationId は無視され、各 AppletId に対応する ProgramId が
 *   自動的に設定されます。
 *
 */
void SetScreenShotAppletIdentityInfo(applet::AppletIdentityInfo info) NN_NOEXCEPT;

//-----------------------------------------------------------------------------

//! @}

//! @name 表示タイミング調整 API
//! @{

/**
    @brief 表示タイミングのハンドリングを有効にします。

    @details
     本関数を呼ぶと、本関数を呼んだアプレットは、foreground になっても絵が自動では表示されなくなり、
     そのタイミングで、システムメッセージ Message_RequestToDisplay が通知されるようになります。

     このメッセージが通知されたら、画面表示の準備を行い、
     準備が完了し、最初の絵を出力したら ApproveToDisplay() を呼ぶことで、
     本関数を呼んだアプレットの絵が表示されるようになります。

     Message_RequestToDisplay は Message_ChangeIntoForeground とは独立して通知されます。
     また Message_RequestToDisplay の通知と同じ回数だけ ApproveToDisplay() を呼ぶ必要があります。
*/
void EnableHandlingForRequestToDisplay() NN_NOEXCEPT;

/**
    @brief 表示タイミングのハンドリングを無効化します。

    @details
     EnableHandlingForRequestToDisplay() での指定を無効化します。
     以後 Message_RequestToDisplay は通知されなくなります。
*/
void DisableHandlingForRequestToDisplay() NN_NOEXCEPT;

/**
    @brief 表示タイミングのハンドリングが有効である際に、画面表示を許可します。

    @see EnableHandlingForRequestToDisplay()
*/
void ApproveToDisplay() NN_NOEXCEPT;

//! @}

//! @name HOME ボタン禁止区間操作
//! @{

/**
    @brief HOME ボタン禁止区間に入ります。

    @see EnterHomeButtonProcessSection()
*/
void EnterHomeButtonProhibitionSection() NN_NOEXCEPT;

/**
    @brief HOME ボタン禁止区間に入ること試行します。

    @return HOME ボタン禁止区間には入れたら true を返し、そうでない場合には false を返します。

    @see EnterHomeButtonProcessSection()
*/
bool TryEnterHomeButtonProhibitionSection() NN_NOEXCEPT;

/**
    @brief HOME ボタン禁止区間を抜けます。

    @see EnterHomeButtonProcessSection()
*/
void LeaveHomeButtonProhibitionSection() NN_NOEXCEPT;

//! @}

//! @name エントランス禁止区間操作
//! @{

/**
    @brief エントランス禁止区間に入ります。

    @see EnterEntranceProcessSection()
*/
void EnterEntranceProhibitionSection() NN_NOEXCEPT;

/**
    @brief エントランス禁止区間に入ること試行します。

    @return エントランス禁止区間には入れたら true を返し、そうでない場合には false を返します。

    @see EnterEntranceProcessSection()
*/
bool TryEnterEntranceProhibitionSection() NN_NOEXCEPT;

/**
    @brief エントランス禁止区間を抜けます。

    @see EnterEntranceProcessSection()
*/
void LeaveEntranceProhibitionSection() NN_NOEXCEPT;

//! @}

//! @name 割込みシーン禁止区間操作
//! @{

/**
    @brief 割込みシーン禁止区間に入ります。

    @see EnterInterruptSceneProcessSection()
*/
void EnterInterruptSceneProhibitionSection() NN_NOEXCEPT;

/**
    @brief 割込みシーン禁止区間に入ること試行します。

    @return 割込みシーン禁止区間には入れたら true を返し、そうでない場合には false を返します。

    @see EnterInterruptSceneProcessSection()
*/
bool TryEnterInterruptSceneProhibitionSection() NN_NOEXCEPT;

/**
    @brief 割込みシーン禁止区間を抜けます。

    @see EnterInterruptSceneProcessSection()
*/
void LeaveInterruptSceneProhibitionSection() NN_NOEXCEPT;

//! @}

//! @name メモリヒープユーティリティ
//! @{

/**
    @brief アプレットの初期化時に os::SetMemoryHeapSize の代わりに呼ぶ関数です。

    @pre
     - size が os::MemoryHeapUnitSize の整数倍である
     - メモリヒープを誰も使用していない

    @post
     - os::GetMemoryHeapSize() == size

    @details
     アプレットの初期化時に os::SetMemoryHeapSize の代わりに本関数を呼んでください。
     本関数は nn::ae ライブラリの初期化外で呼ぶことが可能で、
     nninitStartup 内からの呼び出しが可能です。

     アプレット起動時にはシステムの仕様上、
     本来なら成功するサイズで os::SetMemoryHeapSize() 関数を呼んだ場合に失敗することがあり、
     十分な時間を空けてから再試行する必要があることがあります。
     本関数は失敗時に適切なスレッドスリープを挟んで再試行を試みるユーティリティ関数です。
     なお、何度か再試行を行いますが、一定時間(1 秒から 2 秒程度)であきらめ、エラーを返します。
*/
Result SetMemoryHeapSizeWithRetry(size_t size) NN_NOEXCEPT;

//! @}


//! @name   VR モード関連
//! @{

/**
    @brief  現在が VR モードか否かを返します。

    @return 現在が VR モードなら true を、そうでないなら false を返します。

    @details
     現在が VR モードか否かを bool 値で返します。

     VR モードはシステムの状態として管理され、
     アプリケーションのフォーカス状態に依存しません。

     例えば、アプリケーションが VR モードを有効にした状態で、
     アウトフォーカス状態や BG フォーカス状態となった場合でも、
     本 API は true を返します。

*/
bool IsVrMode() NN_NOEXCEPT;

/**
    @brief VR モード向けの二画面出力に対応しているか否かを宣言します。

    @param[in] isSupported  自アプレットが二画面出力に対応しているか否か

    @pre
     - 自身がライブラリアプレットである
     - nn::ae::InvokeLibraryAppletMain() 呼出し前である

    @details
     本 API は暫定です。
     将来的に仕様を変更する可能性がありますので、ご了承下さい。

     自ライブラリアプレットが VR モード向けの二画面出力に対応しているか否かを
     宣言します。本 API で宣言しなかった場合のデフォルトは false です。

     本 API は必ず nn::ae::InvokeLibraryAppletMain() 呼出し前に発行して下さい。

*/
void SetVrModeDualScreenSupported(bool isSupported) NN_NOEXCEPT;

//! @}

//! @name コントローラ F/W 更新区間の制御
//! @{

/**
    @brief  コントローラ F/W 更新区間中か否かを返します。

    @return コントローラ F/W 更新区間中なら true を、そうでなければ false を返します。

    @details
     コントローラ F/W 更新区間中か否かを bool 値で返します。

*/
bool IsInControllerFirmwareUpdateSection() NN_NOEXCEPT;

//! @}


}} // namespace nn::ae

