﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_Notifiable.h>
#include <nn/account/profile/account_ProfileTypes.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Result.h>
#include <nn/os/os_SdkMutex.h>

namespace nn { namespace account { namespace profile {

enum ProfileStorageTag
{
    ProfileStorageTag_Update,
};

class ProfileStorage
    : detail::Notifiable<ProfileStorageTag, detail::ProfileStorageNotifierCountMax>
{
    NN_DISALLOW_COPY(ProfileStorage);
    NN_DISALLOW_MOVE(ProfileStorage);

private:
    mutable os::SdkMutex m_Lock;
    bool m_Initialized {false};
    const detail::AbstractLocalStorage* m_pStorage {nullptr};

    mutable std::aligned_storage<detail::ProfileIoBufferSize, NN_ALIGNOF(std::max_align_t)>::type m_IoBuffer;
    static_assert(std::alignment_of<decltype(m_IoBuffer)>::value >= std::alignment_of<std::max_align_t>::value, "alignof(m_IoBuffer) < alignof(std::max_align_t)");

    Profile m_Profiles[UserCountMax];
    uint64_t m_Modifier[UserCountMax] {};

    void AddUnsafe(const Uid& uid) NN_NOEXCEPT;
    void DeleteUnsafe(const Uid& uid) NN_NOEXCEPT;
    Result UpdateImplUnsafe(
        const Uid& uid,
        const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size) NN_NOEXCEPT;
    Result ApplyExternalDataUnsafe(
        const Uid& uid,
        const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size,
        bool force) NN_NOEXCEPT;
    void StoreSaveData() const NN_NOEXCEPT;
    Result FindUnsafe(int* pOut, const Uid& uid) const NN_NOEXCEPT;

public:
    ProfileStorage() NN_NOEXCEPT
    {
        for (auto i = 0; i < std::extent<decltype(m_Profiles)>::value; ++ i)
        {
            m_Profiles[i].Clear();
        }
    }

    // アカウントプロセスの初期化/ 終了時に行うべき処理
    Result Initialize(
        const Uid* users, int userCount,
        const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;
    const detail::AbstractLocalStorage& GetStorageRef() NN_NOEXCEPT
    {
        return *m_pStorage;
    }

    // UserRegistration と連動する処理
    void Add(const Uid& uid) NN_NOEXCEPT;
    void Delete(const Uid& uid) NN_NOEXCEPT;

    // ProfileUpdateNotifier からの呼び出し用
    void ReleaseSystemEvent(const os::SystemEventType *pEvent) NN_NOEXCEPT;

    // 本体システム向け
    nn::Result GetProfileUpdateEvent(os::SystemEventType** pOutEvent) NN_NOEXCEPT;

    // Profile や ProfileAdapter で呼ばれる機能の実体
    Result GetDigest(ProfileDigest* pOut, const Uid& uid) const NN_NOEXCEPT;
    Result GetProfile(ProfileBase* pOutProfileBase, UserData* pOutUserData, const Uid& uid) const NN_NOEXCEPT;
    Result GetImageSize(size_t* pOutSize, const Uid& uid) const NN_NOEXCEPT;
    Result LoadImage(size_t* pOutActualSize, void* pOut, size_t size, const Uid& uid) const NN_NOEXCEPT;
    Result Update(const Uid& uid, const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size) NN_NOEXCEPT;
    Result ApplyExternalData(const Uid& uid, const ProfileBase& profileBase, const UserData& userData, const void* pImage, size_t size, bool force) NN_NOEXCEPT;
};

}}} // ~namespace nn::account::profile
