﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/account/detail/account_Execution.h>
#include <nn/http/json/http_RapidJsonInputStream.h>
#include <nn/http/json/http_RapidJsonOutputStream.h>

namespace nn { namespace account { namespace detail {
class AbstractLocalStorage;
}}} // ~namespace nn::account::detail

namespace nn { namespace account { namespace json {

// For input
using StringBufferForRapidJson = nn::http::json::StringBufferForRapidJson;
template <typename Impl>
using BufferedStreamForRapidJson = nn::http::json::BufferedStreamForRapidJson<Impl>;

// For output
using FixedSizeOneTimeAllocatorForRapidJson = nn::http::json::FixedSizeOneTimeAllocatorForRapidJson;
using StringOutputBufferForRapidJson = nn::http::json::StringOutputBufferForRapidJson;

/** @brief ファイル入力を一時的にメモリにプールして IO 効率を向上した InputStreamType の実装です。
    @details
        ImportJsonByRapidJson() の InputStreamType を実装しています。
 */
class BufferedFileInputStreamForRapidJson final
    : public BufferedStreamForRapidJson<BufferedFileInputStreamForRapidJson>
    , public detail::CancelInjectable
{
    NN_DISALLOW_COPY(BufferedFileInputStreamForRapidJson);

private:
    friend nn::http::json::BufferedStreamForRapidJson<BufferedFileInputStreamForRapidJson>;
    size_t FillBufferImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT;

public:
    using Ch = BufferedStreamForRapidJson::Ch;
    NN_STATIC_ASSERT(sizeof(Ch) == 1);

private:
    const detail::AbstractLocalStorage& m_Storage;
    const char* const m_Path;
    size_t m_Offset {0u};
    Result m_IoResult {ResultSuccess()};

public:
    BufferedFileInputStreamForRapidJson(const char* path, const detail::AbstractLocalStorage& storage, detail::Cancellable* pCancellable = nullptr) NN_NOEXCEPT;
    Result GetResult() const NN_NOEXCEPT;
};

class BufferedFileOutputStreamForRapidJson
{
    NN_DISALLOW_COPY(BufferedFileOutputStreamForRapidJson);

public:
    using Ch = char;

private:
    const detail::AbstractLocalStorage& m_Storage;
    const char* const m_Path;
    Ch* const m_Buffer;
    const size_t m_BufferSize;
    size_t m_FilledSize {0u};
    size_t m_FlushSize {0u};
    Result m_IoResult {ResultSuccess()};

    void FlushImpl() NN_NOEXCEPT;

public:
    BufferedFileOutputStreamForRapidJson(const char* path, Ch* buffer, size_t bufferSize, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT;

    Result Finalize(size_t *pOutSize) NN_NOEXCEPT;

    void Put(Ch c) NN_NOEXCEPT;
    void Flush() NN_NOEXCEPT;
};

using DefaultRapidJsonMemoryWriter = nn::http::json::DefaultRapidJsonMemoryWriter;
using DefaultRapidJsonFileWriter = nne::rapidjson::Writer<BufferedFileOutputStreamForRapidJson, nne::rapidjson::UTF8<>, nne::rapidjson::UTF8<>, FixedSizeOneTimeAllocatorForRapidJson>;

}}} // ~namespace nn::account::json
