﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/baas/account_BaasOperator.h>
#include <nn/account/baas/account_BaasLoginCache.h>
#include <nn/account/baas/account_Interface.sfdl.h>
#include <nn/account/detail/account_InternalConfig.h>
#include <nn/account/detail/account_Log.h>
#include <nn/account/detail/account_ReferenceContext.h>
#include <nn/account/detail/account_Settings.h>
#include <nn/account/detail/account_Usage.h>
#include <nn/account/nas/account_NasOperator.h>
#include <nn/account/nas/account_NasSessionPool.h>
#include <nn/account/nas/account_NasUserResourceCache.h>
#include <nn/account/ndas/account_AuthenticationCache.h>
#include <nn/account/ndas/account_NdasOperator.h>
#include <nn/account/profile/account_ProfileStorage.h>
#include <nn/account/user/account_UserRegistry.h>
#include <nn/account/user/account_UserStateManager.h>
#include <nn/account/account_AccountServiceImpl.h>
#include <nn/account/account_IAccountService.sfdl.h>
#include <nn/account/account_RuntimeResource.h>

#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_ISharedObject.h>

#include <nn/srepo/srepo_StateNotifier.h>

namespace nn {
namespace account {
template <typename Allocator, size_t DaemonCount>
class AccountServiceImplWithBackgroundDaemonController;

namespace detail {
class ISessionObject;
} // ~namespace nn::account::detail
} // ~namespace nn::account
}


namespace nn {
namespace account {

enum class BackgroundDaemonControlRequest
{
    Suspend,
    Resume,
};

template <size_t DaemonCount>
class BackgroundDaemonController
{
public:
    typedef void(*RequestHandlerType)(BackgroundDaemonControlRequest request, void* data);

private:
    struct Record
    {
        bool registered;
        RequestHandlerType callback;
        void* data;
    };
    Record m_Records[DaemonCount] {};

    mutable os::SdkMutex m_Lock;
    mutable uint32_t m_Count {0};

    template <BackgroundDaemonControlRequest R>
    NN_FORCEINLINE void OpImpl() const NN_NOEXCEPT
    {
        for (int i = 0; i < DaemonCount && m_Records[i].registered; ++ i)
        {
            m_Records[i].callback(R, m_Records[i].data);
        }
    }

public:
    void Register(RequestHandlerType callback, void* data) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(callback);
        for (auto& record : m_Records)
        {
            if (!record.registered)
            {
                Record r = {true, callback, data};
                record = r;
                return;
            }
        }
        NN_ABORT("[nn::account] No more slot to register daemons: limit=%d\n", DaemonCount);
    }
    void Suspend() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        NN_ABORT_UNLESS(m_Count < std::numeric_limits<decltype(m_Count)>::max());
        if (m_Count ++ == 0)
        {
            NN_DETAIL_ACCOUNT_INFO("[nn::account] Suspending bg daemons\n");
            OpImpl<BackgroundDaemonControlRequest::Suspend>();
            NN_DETAIL_ACCOUNT_INFO("[nn::account] Suspending bg daemons: done\n");
        }
    }
    void Resume() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Lock)> lock(m_Lock);
        NN_ABORT_UNLESS(m_Count > 0);
        if (-- m_Count == 0)
        {
            NN_DETAIL_ACCOUNT_INFO("[nn::account] Resuming bg daemons\n");
            OpImpl<BackgroundDaemonControlRequest::Resume>();
            NN_DETAIL_ACCOUNT_INFO("[nn::account] Resuming bg daemons: done\n");
        }
    }
};

template <typename StorageManager, size_t DaemonCount>
class ServiceResource
{
private:
    /* --------------------------------------------------------
        ServiceFramework 関連
     */
    // UserRegistryImpl 内部で生成する ServiceObject 用のアロケータ
    template <size_t HeapSize>
    class Allocator : public sf::ExpHeapAllocator
    {
    private:
        typedef typename std::aligned_storage<HeapSize>::type StorageType;
        StorageType m_Buffer;
        lmem::HeapHandle m_HeapHandle;

    public:
        typedef sf::ExpHeapAllocator::Policy Policy;

        Allocator() NN_NOEXCEPT
        {
            m_HeapHandle = lmem::CreateExpHeap(&m_Buffer, sizeof(m_Buffer), 0);
            this->Attach(m_HeapHandle);
        }
        ~Allocator() NN_NOEXCEPT
        {
            lmem::DestroyExpHeap(m_HeapHandle);
        }
    };
    typedef Allocator<detail::FactoryHeapSizeMax> AllocatorType;
    typedef sf::ObjectFactory<typename AllocatorType::Policy> FactoryType;
    AllocatorType m_ObjectAllocator;

    sf::EmplacedRef<IAccountServiceForAdministrator, AccountServiceImplWithBackgroundDaemonController<AllocatorType, DaemonCount>> m_pService;
    sf::EmplacedRef<IBaasAccessTokenAccessor, baas::BaasAccessTokenAccessorImpl<AllocatorType>> m_pBaasAccessTokenAccessor;

    /* --------------------------------------------------------
     */
    StorageManager m_Storage;
    user::UserRegistry m_UserRegistry;
    user::UserStateManager m_UserStateManager;
    detail::OpenContextRetainer<AllocatorType> m_OpenContextRetainer;
    profile::ProfileStorage m_ProfileStorage;
    // NDAS
    ndas::ApplicationAuthenticationCache m_AppAuthCache;
    ndas::NdasOperator m_NdasOp;
    // BaaS
    baas::ClientAccessTokenCache m_BaasClientAccessTokenCache;
    baas::UserAccessTokenCache m_BaasUserAccessTokenCache;
    baas::UserIdTokenCache m_BaasUserIdTokenCache;
    baas::BaasUserInfoHolder m_BaasUserInfoHolder;
    baas::BaasOperator m_BaasOp;
    // NAS
    nas::NasUserResourceCache m_NasUserResourceCache;
    nas::NasSessionPool<AllocatorType> m_NasSessionPool;
    nas::NasOperator m_NasOp;

    // 非同期処理用
    Executor m_ExecutorForApplication;
    Executor m_ExecutorForSystem;

    // 利用情報
    detail::Usage m_Usage;

    // BG デーモン制御用
    BackgroundDaemonController<DaemonCount> m_BgDaemonCtrl;

    /* --------------------------------------------------------
     */
    void InitializeImpl() NN_NOEXCEPT;
    Result InitializeUserRegistry() NN_NOEXCEPT;
    Result InitializeProfileStorage(const Uid* users, int userCount) NN_NOEXCEPT;

    void ClearPendings() NN_NOEXCEPT;

public:
    ServiceResource() NN_NOEXCEPT;
    ~ServiceResource() NN_NOEXCEPT;

    template <typename Interface>
    sf::SharedPointer<Interface> GetServicePointer() const NN_NOEXCEPT
    {
        return sf::SharedPointer<Interface>(m_pService);
    }
    sf::SharedPointer<IAccountServiceForApplication> CreateServiceObjectForApplication() NN_NOEXCEPT
    {
        auto p = FactoryType::template CreateSharedEmplaced<IAccountServiceForApplication, AccountServiceImplForApplication<AllocatorType>>(
                &m_ObjectAllocator,
                m_ObjectAllocator, m_Usage,
                m_Storage, m_UserRegistry, m_UserStateManager, m_OpenContextRetainer, m_ProfileStorage, m_NasSessionPool, m_NdasOp, m_BaasOp, m_NasOp, m_ExecutorForApplication);
        NN_ABORT_UNLESS(
            p,
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Failed to create ServiceObjectForApplication (nullptr returned)\n");
        return p;
    }
    sf::SharedPointer<IBaasAccessTokenAccessor> GetBaasAccessTokenAccessorPointer() const NN_NOEXCEPT
    {
        return m_pBaasAccessTokenAccessor;
    }
    Executor& GetExecutorForSystem() NN_NOEXCEPT
    {
        return m_ExecutorForSystem;
    }
    Executor& GetExecutorForApplication() NN_NOEXCEPT
    {
        return m_ExecutorForApplication;
    }
    void RegisterBackgroundDaemon(typename BackgroundDaemonController<DaemonCount>::RequestHandlerType callback, void* data) NN_NOEXCEPT
    {
        m_BgDaemonCtrl.Register(callback, data);
    }
};

}} // ~namespace nn::account


/* --------------------------------------------------------------------------------------------
    実装
 */

#include <nn/account/detail/account_FileSystem.h>
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_SaveData.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace account {

const int DefaultDaemonCount = 1;
typedef ServiceResource<detail::LocalStorage<detail::DefaultFileSystem, detail::SystemSaveDataPolicy::Policy>, DefaultDaemonCount> DefaultServiceResource;

template <typename Allocator, size_t DaemonCount>
class AccountServiceImplWithBackgroundDaemonController
    : public AccountServiceImpl<Allocator>
{
private:
    Allocator& m_Allocator;
    typedef sf::ObjectFactory<typename Allocator::Policy> FactoryType;

    typedef BackgroundDaemonController<DaemonCount> ControllerType;
    const ControllerType& m_BgDaemonCtrl;

    template <typename Suspendable>
    class ScopedSuspension
    {
        const Suspendable& m_Suspendable;
    public:
        explicit ScopedSuspension(const Suspendable& suspendable) NN_NOEXCEPT
            : m_Suspendable(suspendable)
        {
            m_Suspendable.Suspend();
        }
        ~ScopedSuspension() NN_NOEXCEPT
        {
            m_Suspendable.Resume();
        }

        Result Dummy() const NN_NOEXCEPT
        {
            NN_RESULT_THROW(ResultInvalidProtocolAccess());
        }
    };

public:
    template <typename... Types>
    AccountServiceImplWithBackgroundDaemonController(const BackgroundDaemonController<DaemonCount>& bgDaemonCtrl, Allocator& allocator, Types&&... args) NN_NOEXCEPT
        : AccountServiceImpl<Allocator>(allocator, std::forward<Types>(args)...)
        , m_Allocator(allocator)
        , m_BgDaemonCtrl(bgDaemonCtrl)
    {
    }

    Result DeleteUser(const Uid& uid) NN_NOEXCEPT
    {
        ScopedSuspension<decltype(m_BgDaemonCtrl)> s(m_BgDaemonCtrl);
        return AccountServiceImpl<Allocator>::DeleteUser(uid);
    }

    Result SuspendBackgroundDaemon(sf::Out<sf::SharedPointer<detail::ISessionObject>> pOut) NN_NOEXCEPT
    {
        auto p = FactoryType::template CreateSharedEmplaced<detail::ISessionObject, ScopedSuspension<decltype(m_BgDaemonCtrl)>>(
            &m_Allocator, m_BgDaemonCtrl);
        NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionObject());
        *pOut = std::move(p);
        NN_RESULT_SUCCESS;
    }
};

template <typename StorageManager, size_t DaemonCount>
ServiceResource<StorageManager, DaemonCount>::ServiceResource() NN_NOEXCEPT
    : m_NdasOp(m_AppAuthCache)
    , m_BaasOp(
        m_BaasClientAccessTokenCache, m_BaasUserAccessTokenCache, m_BaasUserIdTokenCache, m_BaasUserInfoHolder,
        m_ProfileStorage, m_NdasOp, m_Storage)
    , m_NasSessionPool(m_ObjectAllocator)
    , m_NasOp(m_NasUserResourceCache, m_BaasOp, m_NdasOp, m_Storage)
{
    InitializeImpl();
}

template <typename StorageManager, size_t DaemonCount>
ServiceResource<StorageManager, DaemonCount>::~ServiceResource() NN_NOEXCEPT
{
    {
        auto lock = m_Storage.AcquireWriterLock();
        if (m_pService)
        {
            // 今回作成中のユーザーがいれば、それを削除する
            ClearPendings();
        }
        m_Storage.ClearCache();
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit());
    }
    m_pService.Reset();
    m_pBaasAccessTokenAccessor.Reset();
}

template <typename StorageManager, size_t DaemonCount>
void ServiceResource<StorageManager, DaemonCount>::InitializeImpl() NN_NOEXCEPT
{
    /* --------------------------------------------------------
        最低限のユーザーアカウントサービスの初期化
     */

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // ファームウェア設定の読み込み
    detail::FirmwareSettings::Refresh();
#endif

    // ファイルシステムの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Mount());
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Setup());

    // 本体上のユーザーアカウントの一覧を復元
    NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeUserRegistry());

    /* --------------------------------------------------------
        サブモジュールの初期化
     */
    Uid users[UserCountMax];
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_UserRegistry.ListAllUsers(users, UserCountMax));
    NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeProfileStorage(users, UserCountMax)); // プロフィール
    // NDAS
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_AppAuthCache.Initialize(m_Storage)); // NDAS アプリ認証トークンキャッシュ
    // BaaS
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasClientAccessTokenCache.Initialize(m_Storage)); // BaaS public_client
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasUserAccessTokenCache.Initialize(m_Storage)); // BaaS public_user
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasUserIdTokenCache.Initialize(m_Storage)); // BaaS User ID トークン
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_BaasUserInfoHolder.Initialize(m_NasUserResourceCache, m_Storage));
    // NAS
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_NasUserResourceCache.Initialize(m_Storage)); // NAS User リソース

    /* --------------------------------------------------------
        サービスオブジェクトの初期化
     */
    m_pService = FactoryType::template CreateSharedEmplaced<IAccountServiceForAdministrator, AccountServiceImplWithBackgroundDaemonController<AllocatorType, DaemonCount>>(
            &m_ObjectAllocator,
            m_BgDaemonCtrl, m_ObjectAllocator, m_Usage,
            m_Storage, m_UserRegistry, m_UserStateManager, m_OpenContextRetainer, m_ProfileStorage, m_NasSessionPool, m_NdasOp, m_BaasOp, m_NasOp, m_ExecutorForSystem);
    NN_ABORT_UNLESS(
        m_pService,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Failed to create ServiceObject (nullptr returned)\n");
    m_pBaasAccessTokenAccessor = FactoryType::template CreateSharedEmplaced<IBaasAccessTokenAccessor, baas::BaasAccessTokenAccessorImpl<AllocatorType>>(
            &m_ObjectAllocator,
            m_ObjectAllocator, m_UserRegistry, m_BaasOp, m_ExecutorForSystem);
    NN_ABORT_UNLESS(
        m_pBaasAccessTokenAccessor,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Failed to create BaasAccessTokenAccessor (nullptr returned)\n");

    /* --------------------------------------------------------
        自己診断系
     */
    auto lock = m_Storage.AcquireWriterLock();
    ClearPendings(); // 前回作成中のユーザーがいれば、それを削除する
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Storage.Commit()); // 初期化完了時点でストレージを確定する

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // SystemReportに既存ユーザーを伝える
    nn::srepo::NotifyUserList(users, m_UserRegistry.GetUserCount());
#endif
}

template <typename StorageManager, size_t DaemonCount>
Result ServiceResource<StorageManager, DaemonCount>::InitializeUserRegistry() NN_NOEXCEPT
{
    auto r = m_UserRegistry.Initialize(m_Storage);
    if (ResultSaveDataBroken::Includes(r))
    {
        // ユーザーレジストリのセーブデータが壊れている場合、ユーザーに関するすべての情報を削除する？
        NN_DETAIL_ACCOUNT_ERROR(
            "[nn::account] -----------------------------------------------\n"
            "  ERROR: UserRegistry is broken. Clear all user accounts.\n");
        m_Storage.Clear();
        return m_UserRegistry.Initialize(m_Storage);
    }
    return r;
}

template <typename StorageManager, size_t DaemonCount>
Result ServiceResource<StorageManager, DaemonCount>::InitializeProfileStorage(const Uid* users, int userCount) NN_NOEXCEPT
{
    auto r = m_ProfileStorage.Initialize(users, userCount, m_Storage);
    if (ResultSaveDataBroken::Includes(r))
    {
        // プロフィールの管理ファイルが壊れている場合、全ユーザーのプロフィールを削除する？
        NN_DETAIL_ACCOUNT_ERROR(
            "[nn::account] -----------------------------------------------\n"
            "  ERROR: ProfileStorage is broken. Clear all users' profile.\n");
        m_Storage.ClearProfile();
        return m_ProfileStorage.Initialize(users, userCount, m_Storage);
    }
    return r;
}

template <typename StorageManager, size_t DaemonCount>
void ServiceResource<StorageManager, DaemonCount>::ClearPendings() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pService);
    Uid pendings[1];
    auto count = static_cast<int>(sizeof(pendings) / sizeof(pendings[0]));
    m_UserRegistry.ListPendingUsers(pendings, count);
    for (auto i = 0; i < count && pendings[i]; ++ i)
    {
        NN_DETAIL_ACCOUNT_INFO(
            "[nn::account] Cleaning up incomplete user registration (#%d)\n", i);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pService->CancelUserRegistration(pendings[i]));
    }
}


/* --------------------------------------------------------------------------------------------
 template 特殊化
*/

template <>
class BackgroundDaemonController<0>
{
public:
    typedef void*& RequestHandlerType;

    template <typename RequestHandlerType>
    void Register(RequestHandlerType callback, void* data) NN_NOEXCEPT
    {
        NN_ABORT("[nn::account] Unsupported");
    }
    void Suspend() const NN_NOEXCEPT
    {
        // nop
    }
    void Resume() const NN_NOEXCEPT
    {
        // nop
    }
};

}} // ~namespace nn::account
