﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/account/account_CachedNintendoAccountInfoTypes.h>
#include <nn/account/nas/account_NasUserTypes.h>

namespace nn {
namespace account {
class CachedNintendoAccountInfo;
} // ~namespace nn::account
}

namespace nn {
namespace account {

/**
    @brief 指定したユーザーのネットワークサービスアカウントと連携しているニンテンドーアカウントについて、本体システムにキャッシュされた情報を取得します。

    @param[out] pOut キャッシュされたニンテンドーアカウントの情報の格納先を指すポインタ
    @param[in] handle 対象のユーザーを指す UserHandle
    @param[out] workBuffer 関数の呼び出しに必要なワークバッファ
    @param[in] workBufferSize workBuffer の大きさ (バイト数)

    @retresult
        @handleresult{
            nn::ResultSuccess,
            成功しました。
        }
        @handleresult{
            nn::account::ResultNetworkServiceAccountUnavailable,
            指定したユーザーのネットワークサービスアカウントを利用可能ではありません。
            EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
        }
    @endretresult

    @pre
        - pOut != nullptr
        - workBuffer != nullptr
        - workBufferSize >= nn::account::RequiredBufferSizeForCachedNintendoAccountInfo
    @post
        - *pOut からニンテンドーアカウントの情報のキャッシュを参照できる。

    @details
        指定したユーザーのネットワークサービスアカウントと連携しているニンテンドーアカウントについて、本体システムにキャッシュされた情報を取得します。

        本体システム上のキャッシュは本体システムが必要に応じて更新を行い、アプリケーションがこれを制御することはできません。
        従ってこの関数で取得される CachedNintendoAccountInfo の内容は関数の呼び出しごとに異なる場合があります。

        法的要件などでニンテンドーアカウントの属性値を厳密に検査する必要がある場合は、サーバー上の値を直接参照してください。
*/
Result LoadCachedNintendoAccountInfo(CachedNintendoAccountInfo* pOut, const UserHandle& handle, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
    @brief 本体システムにキャッシュされたニンテンドーアカウントの情報を参照するためのクラスです。
    @details
        ニンテンドーアカウントの各属性がどのような値を取り得るかは、別途ニンテンドーアカウントの仕様を参照してください。
*/
class CachedNintendoAccountInfo
{
    friend Result LoadCachedNintendoAccountInfo(CachedNintendoAccountInfo* pOut, const UserHandle& handle, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

private:
    nas::NasUserBaseForApplication m_Base;

public:
    /**
        @brief 不定な値のオブジェクトを作成します。
        @details
            作成された直後のオブジェクトからは意味のある情報を取得することができません。
            LoadCachedNintendoAccountInfo() 関数に与えることで意味のある情報を取得できるようになります。
    */
    CachedNintendoAccountInfo() NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントに登録された生年月日を表す文字列を取得します。

        @param[out] pOutBytes 返却される文字列の領域の大きさ(バイト数)。終端文字を含みます。

        @return ニンテンドーアカウントに登録された生年月日を表す文字列

        @details
            ニンテンドーアカウントに登録された生年月日を表す文字列を取得します。
            文字列は '\0' で終端されており、終端文字を含む大きさは *pOutBytes バイトです。

            本関数の返す文字列の内容は、ISO 8601の拡張形式に従います。

            この関数が返す領域は *this が破棄されるまでの期間に限り意味のある情報を保持します。
    */
    const char* GetBirthday(size_t* pOutBytes) const NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントに登録された性別を表す文字列を取得します。

        @param[out] pOutBytes 返却される文字列の領域の大きさ(バイト数)。終端文字を含みます。

        @return ニンテンドーアカウントに登録された性別を表す文字列

        @details
            ニンテンドーアカウントに登録された性別を表す文字列を取得します。
            文字列は '\0' で終端されており、終端文字を含む大きさは *pOutBytes バイトです。

            この関数が返す領域は *this が破棄されるまでの期間に限り意味のある情報を保持します。

            返却される文字列の内容については、ニンテンドーアカウントの仕様を参照してください。
    */
    const char* GetGender(size_t* pOutBytes) const NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントに登録された居住国を表す文字列を取得します。

        @param[out] pOutBytes 返却される文字列の領域の大きさ(バイト数)。終端文字を含みます。

        @return ニンテンドーアカウントに登録された居住国を表す文字列

        @details
            ニンテンドーアカウントに登録された居住国を表す文字列を取得します。
            文字列は '\0' で終端されており、終端文字を含む大きさは *pOutBytes バイトです。

            本関数の返す文字列の内容は、(実装時点ではなく)関数呼び出し時点でのISO 3166-1 alpha-2に準拠します。
            従ってこの文字列を nn::util::GetCountryFromIso31661Alpha2() に与えると失敗する場合があることに注意してください。

            この関数が返す領域は *this が破棄されるまでの期間に限り意味のある情報を保持します。
    */
    const char* GetCountry(size_t* pOutBytes) const NN_NOEXCEPT;
};

} // ~namespace nn::account
}
