﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/** @file
    @brief  アカウントシステムが特権プログラム向けに限定して提供する API を宣言します。
 */

#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_Declaration.h>
#include <nn/account/account_ConfigForSystemServices.h>
#include <nn/account/account_OAuthProcedure.h>
#include <nn/account/account_Types.h>
#include <nn/account/profile/account_ProfileTypes.h>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace nn { namespace account {
namespace baas {
class IAdministrator;
} // ~namespace nn::account::baas

namespace profile{
class IProfileEditor;
} // ~namespace nn::account::profile

class NetworkServiceAccountAdministrator;
class ProfileEditor;
}} // ~namespace nn::account

namespace nn { namespace account {

//! @name アカウントシステムの管理機能を提供する関数
//! @{

/**
    @brief アカウントシステムを、管理者権限をもって初期化します。

    @pre
        - 本 API 以外の手段でアカウントシステムを初期化していない
    @post
        - アカウントシステムが提供する機能のうち、管理者向けの機能を使用可能

    @details
        アカウントシステムの初期化処理を行い、利用可能な状態にします。
        この関数はアカウントシステムを利用するにプロセスつき、少なくとも 1 度呼び出す必要があります。
        アカウントシステムはプロセスごとに初期化処理の呼出し回数を管理しています。
        同一プロセスでの重複する初期化処理の呼出しではこの回数の計数のみを行い、実際の初期化処理は行いません。

        初期化回数は Initialize() 関数など、本 API 以外の初期化手段の呼び出しも含めて計数されます。
        従って本関数より先に別の初期化手段を実行すると、本関数の呼び出しで利用可能になる追加の機能は使用できません。

        管理者権限を持たないプロセスによる本関数の呼び出しは失敗します。
*/
void InitializeForAdministrator() NN_NOEXCEPT;

/**
    @brief ユーザーの新規追加を開始します。

    @param[out] pOutUser 新規追加を開始したユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultRegistrationAlreadyBegins,
            ユーザーの新規追加が既に実行中です。
        }
        @handleresult{
            nn::account::ResultUserRegistryFull,
            これ以上ユーザーを追加することができません。
        }
    @endretresult

    @pre
        - ユーザーの新規追加が実行中でない
        - アカウントシステムに存在するユーザーの数が UserCountMax より小さい
    @post
        - ユーザーの新規追加が実行中
        - static_cast<bool>(*pOutUser) == true

    @details
        アカウントシステムへの新しいユーザーの追加を開始します。
        ユーザーの新規追加は CompleteUserRegistration() を呼び出すことで確定します。
        未完了のユーザーの新規追加は、アカウントシステムの終了時点で取り消され、システムの再起動を跨ぐことはありません。
        また、 CancelUserRegistration() の呼び出しで新規追加を明示的に取り消すことができます。

        追加が未完了のユーザーは、リストアップされずOpenやCloseの状態を持ちませんが、次の操作が可能です。
        - プロフィールの変更

        同時に新規追加を開始可能なユーザーの数は 1 です。
*/
Result BeginUserRegistration(Uid* pOutUser) NN_NOEXCEPT;

/**
    @brief ユーザーの新規追加を完了します。

    @param[in] user 新規追加を実行中のユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            指定したユーザーが作成中ではありません。
        }
        @handleresult{
            nn::account::ResultRegistrationNotPermitted,
            nn::account::Initialize() を呼び出したプログラムが実行中のため、ユーザーアカウントの追加が許可されません。
            CancelUserRegistration() で中止するか、 CompleteUserRegistrationForcibly() で強制的に完了できます。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - user が指すユーザーが新規作成中
        - nn::account::Initialize を呼び出したプログラムが実行中
    @post
        - ユーザーの新規追加が実行中でない
        - user が指すユーザーは Closed 状態で存在

    @details
        アカウントシステムへの新しいユーザーの追加を完了します。

        新しく追加されたユーザーは次の性質をもちます。
        - Closed 状態である。
        - ListAllUser ではリストの最後尾に位置する。
*/
Result CompleteUserRegistration(const Uid& user) NN_NOEXCEPT;

/**
    @brief ユーザーの新規追加を強制的に完了します。

    @param[in] user 新規追加を実行中のユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            指定したユーザーが作成中ではありません。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - user が指すユーザーが新規作成中
    @post
        - ユーザーの新規追加が実行中でない
        - user が指すユーザーは Closed 状態で存在

    @details
        アカウントシステムへの新しいユーザーの追加を完了します。
        本関数は CompleteUserRegistration() が nn::account::ResultRegistrationNotPermitted を返す場合でも成功します。

        新しく追加されたユーザーは次の性質をもちます。
        - Closed 状態である。
        - ListAllUser ではリストの最後尾に位置する。
*/
Result CompleteUserRegistrationForcibly(const Uid& user) NN_NOEXCEPT;

/**
    @brief ユーザーの新規追加を取り消します。

    @param[in] user 新規追加を実行中のユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            指定したユーザーが作成中ではありません。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - user が指すユーザーが新規作成中
    @post
        - ユーザーの新規追加が実行中でない
        - user が指すユーザーが不在

    @details
        アカウントシステムへの新しいユーザーの追加を取り消します。
*/
Result CancelUserRegistration(const Uid& user) NN_NOEXCEPT;

/**
    @brief ユーザーアカウントを削除します。

    @param[in] user 対象のユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            対象のユーザーアカウントが存在しません。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - user が指すユーザーが Closed 状態で存在
    @post
        - user が指すユーザーが不在

    @details
        アカウントシステム上の既存のユーザーを削除します。
        当該ユーザーに関してアカウントシステムが保有する情報はすべて破棄されます。
*/
Result DeleteUser(const Uid& user) NN_NOEXCEPT;

/**
    @brief ユーザーの、全ユーザーのリスト上での位置を変更します。

    @param[in] user 対象のユーザーを指す Uid
    @param[in] position 対象のユーザーの、全ユーザーのリスト中の新しい位置

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            対象のユーザーアカウントが存在しません。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - user が指すユーザーが存在
        - position >= 0 && position < "全ユーザー数"
    @post
        - ListAllUsers() で取得するユーザーリストLについて、L[position] == user
        - '対象のユーザーの元の位置p' < position のとき、 (p, position] のユーザーはそれぞれ前方に 1 移動
        - '対象のユーザーの元の位置p' > position のとき、 [position, p) のユーザーはそれぞれ後方に 1 移動

    @details
        ListAllUsers() で得られるユーザーリストでの、 user が指すユーザーの位置を変更します。
        対象のユーザーは、インデックスが 0 始まりのリストで position 番目に挿入されます。

        この位置の変更は、 ListOpenUsers() 関数の結果にも影響します。
*/
Result SetUserPosition(const Uid& user, int position) NN_NOEXCEPT;

/**
    @brief 指定したユーザーに関して ProfileEditor オブジェクトを取得します。

    @param[out] pOutProfile 取得した ProfileEditor オブジェクトの格納先
    @param[in] user 対象のユーザーを指す Uid

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
        @handleresult{
            nn::account::ResultUserNotExist,
            対象のユーザーアカウントが存在しません。
        }
    @endretresult

    @pre
        - static_cast<bool>(user) == true
        - pOutProfile != nullptr
        - 対象のユーザーが存在するか新規作成中
    @post
        - *pOutProfile を使用して対象ユーザーのプロフィール情報を編集可能

    @details
        対象のユーザーのプロフィールを変更するための ProfileEditor オブジェクトを取得します。
        この関数で取得した ProfileEditor オブジェクトには、取得時点でのプロフィールが既にキャッシュされています。
*/
Result GetProfileEditor(ProfileEditor* pOutProfile, const Uid& user) NN_NOEXCEPT;

/**
    @brief インターネットとの通信を伴う機能を利用するための、アカウントシステム単位の認証を明示的に行います。

    @param[out] pOutContext 非同期処理の完了を通知するためのオブジェクト

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次の関数で初期化済み
            - InitializeForAdministrator()
        - pOut != nullptr
        - インターネットと通信可能
    @post
        - アカウントシステムの、インターネットとの通信を伴う機能を利用可能

    @details
        インターネットとの通信を伴う機能を利用するための、アカウントシステム単位の認証を明示的に行います。
        これによって呼び出し元の任意のタイミングでシステム単位の異常を検出できます。

        通常この認証は他の機能の利用に際して暗黙的に行われます。
        従って、システム単位の異常を意図的に確認する場合を除き、本関数を使用する必要はありません。
        また本関数が成功した場合でも、各機能が別個に行う認証処理において異常が検出されることがあります。

        この関数はインターネットとの通信を行います。

        この関数は即時返り、実際の処理は非同期的に実行されます。
        処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。

        非同期処理の結果として本体システムの更新が必要であることを検知した場合、 pOutContext->GetResult() は nn::account::ResultUnacceptableSystemVersion を返却します。
*/
Result AuthenticateServiceAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

/**
    @brief 指定したユーザーのネットワークサービスアカウントを管理する NetworkServiceAccountAdministrator オブジェクトを取得します。

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理が成功しました。
        }
    @endretresult

    @pre
        - ライブラリが次の関数で初期化済み
            - InitializeForAdministrator()
        - pOut != nullptr
        - uid が指すユーザーが存在
    @post
        - *pOut が、対象のユーザーのネットワークサービスアカウントを参照している

    @details
        ユーザーを指定して、そのユーザーのネットワークサービスアカウントを管理する NetworkServiceAccountAdministrator オブジェクトを取得します。
        NetworkServiceAccountAdministrator オブジェクトを使用すると、ネットワークサービスアカウントの情報を管理者権限で取り扱うことができます。
*/
Result GetNetworkServiceAccountAdministrator(NetworkServiceAccountAdministrator* pOut, const Uid& uid) NN_NOEXCEPT;

/**
    @brief アカウントシステム外部のネットワークサービスアカウントをアカウントシステムに登録可能にするための手続きオブジェクトを取得します。

    @param[out] pOut 手続きを実施するためのオブジェクト
    @param[in] sessionId 手続きオブジェクトを指すセッションID

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
    @endretresult

    @pre
        - 初期化状態
        - pOut != nullptr
    @post
        - *pOut が手続きを実施可能

    @details
        アカウントシステム外部のネットワークサービスアカウントをアカウントシステムに登録可能するための手続きオブジェクトを取得します。
        この手続きでは、認証や認可の結果は ExternalNetworkServiceAccountIntroducingProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
        従って ExternalNetworkServiceAccountIntroducingProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。

        この手続きの完了時点でネットワークサービスアカウントはアカウントシステムに登録可能な状態になりますが、対応するユーザーアカウントは作成されません。
        対応するユーザーアカウントを作成するまでにシステムが終了した場合、ネットワークサービスアカウントの登録は中止されます。
*/
Result ProxyProcedureToIntroduceExternalNetworkServiceAccountForRegistration(ExternalNetworkServiceAccountIntroducingProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

/**
    @brief アカウントシステム外部のネットワークサービスアカウントを利用可能にするための手続きオブジェクトを取得します。

    @param[out] pOut 手続きを実施するためのオブジェクト
    @param[in] sessionId 手続きオブジェクトを指すセッションID

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
    @endretresult

    @pre
        - 初期化状態
        - pOut != nullptr
    @post
        - *pOut が手続きを実施可能

    @details
        アカウントシステム外部のネットワークサービスアカウントを利用可能にするための手続きオブジェクトを取得します。
        この手続きでは、認証や認可の結果は ExternalNetworkServiceAccountIntroducingProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
        従って ExternalNetworkServiceAccountIntroducingProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。

        この手続きの完了時点で、要求元のプログラムでネットワークサービスアカウントの認証トークンを一時的に利用可能になります。
        このとき、対応するユーザーアカウントは作成されません。
*/
Result ProxyProcedureToIntroduceExternalNetworkServiceAccount(ExternalNetworkServiceAccountIntroducingProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

//! @}

/**
    @brief ユーザーのプロフィールを変更するためのクラスです。

    @details
        ProfileEditor クラスは、オブジェクト初期化時に関連付けたユーザーのプロフィールを変更する機能を提供します。
        ユーザーのプロフィールは次の粒度で変更することができます。
        - ニックネーム
        - 任意データ
        - プロフィール画像

        ProfileEditor オブジェクトは内部にプロフィールをキャッシュし、プロフィール画像以外のプロフィール情報の取得や変更はこのキャッシュを経由して行います。
        ProfileEditor オブジェクト上でプロフィールを変更したあと、それをアカウントシステムに反映するには Flush() や FlushWithImage() を呼びます。
        このキャッシュを破棄し、アカウントシステムからオリジナルの情報を再取得するには Refresh() を呼びます。

        ProfileEditor オブジェクトを GetProfileEditor() 関数で取得する場合、取得した時点でのプロフィールがキャッシュされています。
*/
class ProfileEditor
{
    NN_DISALLOW_COPY(ProfileEditor);
    friend Result GetProfileEditor(ProfileEditor* pOutProfile, const Uid& user) NN_NOEXCEPT;

private:
    Uid m_Uid;
    profile::IProfileEditor* m_Ptr;
    profile::ProfileBase m_Base;
    profile::UserData m_UserData;

    ProfileEditor& Swap(ProfileEditor& rhs) NN_NOEXCEPT;
    explicit ProfileEditor(const Uid& user, profile::IProfileEditor* ptr) NN_NOEXCEPT;
    Result GetProfileImageSize(size_t* pOutSize) const NN_NOEXCEPT;

public:
    /**
        @brief ProfileEditor クラスのデフォルトコンストラクタ

        @post
            - *this はどのユーザーとも関連づかない

        @details
            ProfileEditor クラスのオブジェクトを、どのユーザーとも関連付けずに作成します。
            この状態ではプロフィールを変更できません。
    */
    ProfileEditor() NN_NOEXCEPT;
    /**
        @brief ProfileEditor クラスのムーブコンストラクタ

        @param[in] rhs *this に内容を引き継ぐ対象の ProfileEditor オブジェクト

        @post
            - *this は rhs の所有していたユーザーとの関連付けを所有
            - rhs はどのユーザーとも関連づかない

        @details
            右辺に指定された ProfileEditor オブジェクトの内容を引き継いで、新しい ProfileEditor オブジェクトを作成します。
            このとき右辺の内容は ProfileEditor() を呼ばれた場合と同等になります。
    */
    ProfileEditor(ProfileEditor&& rhs) NN_NOEXCEPT;
    /**
        @brief ProfileEditor クラスのデストラクタ

        @details
            ProfileEditor オブジェクトを破棄します。
            ProfileEditor オブジェクトがユーザーと関連付くとき、アカウントシステムに未反映の変更内容は破棄されます。
    */
    virtual ~ProfileEditor() NN_NOEXCEPT;
    /**
        @brief ProfileEditor クラスのムーブ代入演算子

        @param[in] rhs *this に内容を引き継ぐ対象の ProfileEditor オブジェクト

        @return *this

        @post
            - *this が所有していた、ユーザーとの関連付けの破棄
            - *this は rhs が所有していた関連付けを所有
            - rhs はどのユーザーとも関連付かない。

        @details
            左辺が事前に保持していた情報を、右辺の内容で置き換えます。
            また右辺の内容は ProfileEditor() を呼ばれた場合と同等になります。
    */
    ProfileEditor& operator=(ProfileEditor&& rhs) NN_NOEXCEPT;

    /**
        @brief キャッシュを再構築します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this がユーザーと関連づく
        @post
            - *this がキャッシュするプロフィール情報が、アカウントシステムの状態と一致

        @details
            ProfileEditor が保持するユーザーのプロフィール情報のキャッシュが、アカウントシステムの最新の状態に基づいて再構築されます。
    */
    Result Refresh() NN_NOEXCEPT;

    /**
        @brief ニックネームを取得します。

        @param[out] pOut 取得したニックネームの格納先

        @pre
            - *this がユーザーと関連づく
            - pOut != nullptr
        @post
            - pOut->name[0, NicknameBytesMax] にひとつ以上の '\0' が存在

        @details
            ProfileEditor がキャッシュするユーザーのプロフィール情報のうち、ニックネームを取得します。
            ユーザーが作成されたばかりでプロフィールが設定されていない場合、次の有効な値を返します。
            - pOUt->name[0] == '\0'
    */
    void GetNickname(Nickname* pOut) const NN_NOEXCEPT;
    /**
        @brief 任意データを取得します。

        @param[out] outUserData 取得したユーザーデータの格納先
        @param[in] bytes pOut に書き込み可能な大きさ (バイト数)

        @pre
            - *this がユーザーと関連づく
            - outUserData != nullptr
            - bytes == nn::accout::UserDataBytesMax

        @details
            ProfileEditor がキャッシュするユーザーのプロフィール情報のうち、任意データを取得します。
            任意データとは、本体機能がユーザーごとに設定する nn::accout::UserDataBytesMax バイトのバイナリデータを指します。
            アカウントシステムはこの内容について関知や改変を行いません。
    */
    void GetUserData(char* outUserData, size_t bytes) const NN_NOEXCEPT;
    /**
        @brief プロフィール画像を取得します。

        @param[out] pOutActualSize プロフィール画像の実際の大きさ (バイト数) の格納先
        @param[out] outImage プロフィール画像の格納先
        @param[in] bufferSize outImage に書き込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this がユーザーと関連づく
            - pOutActualSize != nullptr
        @post
            - *pOutActualSize が画像データの実際の大きさ (バイト数) を保持
            - outImage != nullptr のとき、 outImage[0, min(*pOutActualSize, bufferSize)) が画像データを保持
            - outImage の保持する画像データをJPEGとして解釈可能

        @details
            ProfileEditor オブジェクトに紐づくユーザーのプロフィール画像を取得します。
            第 2 引数に nullptr を指定することで、プロフィール画像の大きさ (バイト数) のみを取得できます。

            プロフィール画像は下記の形式に従います。
            - 画像解像度は縦横各256px
            - 色空間はsRGB
            - ベースライン方式のJPEGフォーマット
                - YCbCr サンプリング比は 4:2:0, 4:2:2, 4:4:4 のいずれか
            - 大きさ (バイト数) は128KiB (131,072バイト) を上限とする
    */
    Result LoadProfileImage(size_t* pOutActualSize, void* outImage, size_t bufferSize) const NN_NOEXCEPT;;

    /**
        @brief ニックネームを設定します。

        @param[in] in 設定するニックネーム

        @pre
            - *this がユーザーと関連づく
            - pOut->name[0, NicknameBytesMax] にひとつ以上の '\0' が存在
        @post
            - 直後の GetNickname() の呼び出しが in と等しい内容を返却

        @details
            ProfileEditor がキャッシュするユーザーのニックネームを、指定したもので置き換えます。

            このメソッド呼び出しでプロフィールに加えた変更はキャッシュに留まります。
            変更をアカウントシステムに反映するには Flush() や FlushWithImage() を呼び出します。
    */
    void SetNickname(const Nickname& in) NN_NOEXCEPT;
    /**
        @brief 任意データを設定します。

        @param[in] userData 設定するユーザーデータ
        @param[in] bytes userData から読み込み可能な大きさ (バイト数)

        @pre
            - *this がユーザーと関連づく
            - userData != nullptr
            - bytes == nn::accout::UserDataBytesMax
        @post
            - 直後の GetUserData() 呼び出しが userData と等しい内容を返却

        @details
            ProfileEditor がキャッシュするユーザーの任意データを、指定したもので置き換えます。
            任意データとは、本体機能がユーザーごとに設定する nn::accout::UserDataBytesMax バイトのバイナリデータを指します。
            アカウントシステムはこの内容について関知や改変を行いません。

            このメソッド呼び出しでプロフィールに加えた変更はキャッシュに留まります。
            変更をアカウントシステムに反映するには Flush() や FlushWithImage() を呼び出します。
    */
    void SetUserData(const char* userData, size_t bytes) NN_NOEXCEPT;

    /**
        @brief キャッシュの内容をアカウントシステムに適用します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this がユーザーと関連づく
        @post
            - *this がキャッシュするプロフィール情報が、アカウントシステムの状態と一致

        @details
            ProfileEditor がキャッシュするユーザーのプロフィールをアカウントシステムに反映します。
            SetNickname() や SetUserData() で変更したプロフィールを保存するにはこのメソッドか FlushWithImage() を呼びます。
    */
    Result Flush() NN_NOEXCEPT;
    /**
        @brief キャッシュの内容を、新しいプロフィール画像とともにアカウントシステムに適用します。

        @param[in] image 適用するプロフィール画像の先頭を指すアドレス
        @param[in] imageSize image から読み込み可能な大きさ (バイト数)

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this がユーザーと関連づく
            - image != nullptr
            - image の保持する画像データをJPEGとして解釈可能
            - imageSize > 0
        @post
            - *this がキャッシュするプロフィール情報が、アカウントシステムの状態と一致
            - 直後の LoadProfileImage() の呼び出しが image と等しい内容を返却

        @details
            ProfileEditor がキャッシュするユーザーのプロフィールをアカウントシステムに反映します。
            SetNickname() や SetUserData() で変更したプロフィールを保存するにはこのメソッドか Flush() を呼びます。

            このメソッドではキャッシュの適用と併せて、引数に指定したプロフィール画像をユーザー向けに保存します。

            プロフィール画像は下記の形式に従う必要があります。
            ただしこのメソッドはその正当性を評価しません。
            - 画像解像度は縦横各256px
            - 色空間はsRGB
            - ベースライン方式のJPEGフォーマット
                - YCbCr サンプリング比は 4:2:0, 4:2:2, 4:4:4 のいずれか
            - 大きさ (バイト数) は128KiB (131,072バイト) を上限とする
    */
    Result FlushWithImage(const void* image, size_t imageSize) NN_NOEXCEPT;

    // 内部向け
    bool IsLocalOrigin() const NN_NOEXCEPT;
    uint64_t GetTimeStamp() const NN_NOEXCEPT;
};

/**
    @brief ユーザーのネットワークサービスアカウントを管理するためのクラスです。

    @details
        アカウントシステムがユーザーごとに管理するネットワークサービスアカウントの情報やトークンのキャッシュを操作するためのクラスです。

        NetworkServiceAccountAdministrator オブジェクトは次の状態を持ちます。
        - 無効
            - オブジェクトはいかなるユーザーのネットワークサービスアカウントとも関連づきません。
            - この状態では、デストラクタ以外を呼び出すことはできません。
        - 有効
            - このオブジェクトは特定のユーザーのネットワークサービスアカウントと関連づきます。
            - この状態では関連づくネットワークサービスアカウントに関して先述の操作が可能です。

        無効状態のインスタンスを GetNetworkServiceAccountAdministrator() に与えることで、 GetNetworkServiceAccountAdministrator() に指定したユーザーのネットワークサービスアカウントを参照して初期化されます。
        また、有効状態のインスタンスを再度 GetNetworkServiceAccountAdministrator() に与えることで、別のユーザーのネットワークサービスアカウントを参照することができます。
*/
class NetworkServiceAccountAdministrator final
    : public NetworkServiceAccountManager
{
    NN_DISALLOW_COPY(NetworkServiceAccountAdministrator);
    NN_DISALLOW_MOVE(NetworkServiceAccountAdministrator);
    friend Result GetNetworkServiceAccountAdministrator(NetworkServiceAccountAdministrator* pOut, const Uid& uid) NN_NOEXCEPT;

protected:
    NetworkServiceAccountAdministrator& Swap(NetworkServiceAccountAdministrator& rhs) NN_NOEXCEPT;
    NetworkServiceAccountAdministrator(const Uid& user, baas::IAdministrator* ptr) NN_NOEXCEPT;

    baas::IAdministrator* GetPtr() const NN_NOEXCEPT
    {
        return reinterpret_cast<baas::IAdministrator*>(NetworkServiceAccountManager::GetPtr());
    }

public:
    /**
        @brief 無効なインスタンスを作成するデフォルトコンストラクタ

        @post
            - *this は無効

        @details
            インスタンスを無効な状態で作成します。
            この状態では、デストラクタ以外のいかなる API も呼出すことができません。

            インスタンスを有効な状態にするには、 GetNetworkServiceAccountAdministrator() を使用してください。
    */
    NetworkServiceAccountAdministrator() NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントを新たに作成します。

        @param[out] pOutContext 非同期処理の完了を通知するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountAlreadyRegistered,
                このユーザーアカウントにはすでにネットワークサービスアカウントが作成されています。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutContext != nullptr
            - *pOutContext が実行中の非同期処理と関連づかない
            - IsNetworkServiceAccountRegistered() が false を返す
            - インターネットと通信可能
        @post
            - IsNetworkServiceAccountRegistered() が true を返す

        @details
            ネットワークサービスアカウントを新たに作成します。
            ネットワークサービスアカウントはフレンドシステム (nn::friends) や NEX の機能を使用する際に必要となります。

            この関数はインターネットとの通信を行います。

            この関数は即時返り、実際の処理は非同期的に実行されます。
            処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。
    */
    Result RegisterAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief 本体システムからネットワークサービスアカウントの登録情報を削除します。

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this が有効
            - IsNetworkServiceAccountRegistered() が true を返す
        @post
            - IsNetworkServiceAccountRegistered() が false を返す

        @details
            指定したユーザーアカウントにおけるネットワークサービスアカウントの登録情報を本体システムから削除します。
            ネットワーク上のサーバーからはいかなる情報も削除されません。

            本関数は nn::ns::UnregisterNetworkServiceAccount() 関数を利用できない状況において、ネットワークサービスアカウントの登録情報をやむを得ず削除するために使用することを想定しています。
            ネットワークサービスアカウントの登録を解除する場合、通常は nn::ns::UnregisterNetworkServiceAccount() 関数を使用してください。

            事前にニンテンドーアカウントとの連携が完了している場合、ユーザーのニンテンドーアカウントを参照できなくなります。
    */
    Result DeleteRegistrationInfoLocally() NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントが登録済みかを検査します。

        @param[out] pOut ネットワークサービスアカウントが登録済みかどうか

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this が有効
            - pOut != nullptr
        @post
            - 対象のユーザーがネットワークサービスアカウントが登録済みの場合に限り *pOut = true

        @details
            ネットワークサービスアカウントが登録済みかを検査します。
            登録済みでない場合、 RegisterAsync() を呼ぶことで登録できます。
    */
    Result IsNetworkServiceAccountRegistered(bool* pOut) NN_NOEXCEPT;

    /**
        @brief ユーザーのプロフィールをネットワークサービスアカウント上のキャッシュと同期します。

        @param[out] pOutContext 非同期処理の完了を通知するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutContext != nullptr
            - *pOutContext が実行中の非同期処理と関連づかない
            - IsNetworkServiceAccountAvailable() が true を返す
            - インターネットと通信可能
        @post
            - 対象のユーザーのプロフィールが変更されていることがある

        @details
            ユーザーのプロフィール情報をネットワークサービスアカウント上のキャッシュと同期します。

            ネットワークサービスアカウントは、その関連づくユーザーのプロフィール情報をキャッシュします。
            このプロフィール情報のキャッシュは、デバイスによらずすべての関連づくユーザーアカウント間で共有され、それらのユーザーアカウントは可能な限り同一のプロフィール情報をもつことが期待されます。

            本関数は、現在のデバイスでのユーザーのプロフィール情報とネットワークサービスアカウント上のキャッシュを、明示的に最新の状態に同期します。
            従って別のデバイスでプロフィール情報が編集されている場合、これがネットワークサービスアカウント上のキャッシュを経由して現在のデバイスのユーザーアカウントに反映されます。
            一方で現在のデバイスでユーザーのプロフィール情報を編集した場合、別のデバイスのユーザーアカウントは編集後のプロフィール情報で上書きされるようになります。

            ふたつのデバイスで同時期にプロフィール情報が編集された場合、おおよその場合で新しいと判断される編集内容が適用され、古いと判断される編集内容は破棄されます。

            この関数はインターネットとの通信を行います。

            この関数は即時返り、実際の処理は非同期的に実行されます。
            処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。
    */
    Result SynchronizeProfileAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief ユーザーのプロフィールでネットワークサービスアカウント上のキャッシュを更新します。

        @param[out] pOutContext 非同期処理の完了を通知するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - pOutContext != nullptr
            - *pOutContext が実行中の非同期処理と関連づかない
            - IsNetworkServiceAccountAvailable() が true を返す
            - インターネットと通信可能
        @post
            - 対象のユーザーのプロフィールが変更されていることがある

        @details
            本関数は、指定したユーザーアカウントのプロフィール情報で、ネットワークサービスアカウント上のキャッシュを明示的に上書きします。
            キャッシュが上書きされるのみであり、関連付くすべてのユーザーアカウントに対し強制的にプロフィール情報を反映するわけではないことに注意してください。

            この関数はインターネットとの通信を行います。

            この関数は即時返り、実際の処理は非同期的に実行されます。
            処理の完了の通知は、 *pOutContext から取得可能な nn::os::SystemEvent のシグナルによって行われます。
    */
    Result UploadProfileAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントがニンテンドーアカウントと連携しているかを検査します。

        @param[out] pOut 連携しているニンテンドーアカウントの有無

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - *this が有効
            - pOut != nullptr
        @post
            - 対象のユーザーのネットワークサービスアカウントがニンテンドーアカウントと連携している場合に限り *pOut = true

        @details
            ネットワークサービスアカウントがニンテンドーアカウントと連携しているかを検査します。
    */
    Result IsLinkedWithNintendoAccount(bool* pOut) NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントとネットワークサービスアカウントを連携する手続きのためのオブジェクトを取得します。

        @param[out] pOut        手続きを実施するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
            @handleresult{
                nn::account::ResultNintendoAccountAlreadyLinked,
                ニンテンドーアカウントとネットワークサービスアカウントがすでに連携済みです。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountRegistered() が true を返す
            - IsLinkedWithNintendoAccount() が false を返す
        @post
            - *pOut が連携の手続きを実施可能

        @details
            ネットワークサービスアカウントとニンテンドーアカウントを連携する手続きのためのオブジェクトを取得します。
            ニンテンドーアカウントとの連携には、ユーザーによる認証及び認可の操作が必要です。
            この手続きでは、認証や認可の結果は NintendoAccountLinkageProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
            従って NintendoAccountLinkageProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。

            連携の手続きが成功すると、ユーザーのネットワークサービスアカウントが過去使用されたことのあるネットワークサービスアカウントで置き換えられる可能性があります。
            ユーザーのネットワークサービスアカウントが過去使用されたことのあるネットワークサービスアカウントで置き換えられた場合、ユーザーアカウントの次の属性が変更されます。
            - ユーザーのネットワークサービスアカウントIDは、過去使用されたことのあるネットワークサービスアカウントのIDで置き換えられます。
            - ユーザーのプロフィールは、過去使用されたことのあるネットワークサービスアカウントにキャッシュされたプロフィールで置き換えられます。
                - 従って、現在ユーザーアカウントに保存されているプロフィール情報は破棄されます。

            この手続きが完了すると、同一のニンテンドーアカウントに対する次回以降の本関数の呼出しにおいて、先述のネットワークサービスアカウントの置き換えが発生するようになります。
            これは、この手続きで指定するニンテンドーアカウントに対し、ネットワークサービスアカウントが永続化されることを意味します。
    */
    Result CreateProcedureToLinkWithNintendoAccount(NintendoAccountLinkageProcedure* pOut) NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントとネットワークサービスアカウントを連携する手続きを再開します。

        @param[out] pOut        手続きを実施するためのオブジェクト
        @param[in] sessionId    OAuthProcedure::Suspend() によって取得したセッション識別子

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountRegistered() が true を返す
            - IsLinkedWithNintendoAccount() が false を返す
        @post
            - *pOut が連携の手続きを実施可能

        @details
            ネットワークサービスアカウントとニンテンドーアカウントを連携する手続きを再開します。
            再開された手続きにおいて OAuthProcedure::GetRequest() を使用することはできません。
    */
    Result ResumeProcedureToLinkWithNintendoAccount(NintendoAccountLinkageProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

    /**
        @brief ネットワークサービスアカウントとニンテンドーアカウントの連携状態を更新する手続きのためのオブジェクトを取得します。

        @param[out] pOut        手続きを実施するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountRegistered() が true を返す
            - IsLinkedWithNintendoAccount() が true を返す
        @post
            - *pOut が同意状態を更新する手続きを実施可能

        @details
            ネットワークサービスアカウントとニンテンドーアカウントの連携状態を更新する手続きのためのオブジェクトを取得します。
            ニンテンドーアカウントとの連携状態の更新には、ユーザーによる認証及び認可の操作が必要です。
            この手続きでは、認証や認可の結果は NintendoAccountLinkageStateUpdateProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
            従って NintendoAccountLinkageStateUpdateProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。

            連携状態の更新は次の場合に必要です。
            - EULAやプライバシーポリシーが更新され、再同意が必要な場合
            - 前回の認証処理から一定の期間が経過した場合
            - その他確立済みのセッションが無効になった場合
    */
    Result CreateProcedureToUpdateLinkageStateOfNintendoAccount(NintendoAccountLinkageStateUpdateProcedure* pOut) NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントとネットワークサービスアカウントの連携状態を更新する手続きを再開します。

        @param[out] pOut        手続きを実施するためのオブジェクト
        @param[in] sessionId    OAuthProcedure::Suspend() によって取得したセッション識別子

        @retresult
            @handleresult{
                nn::ResultSuccess,
                再開に成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountRegistered() が true を返す
            - IsLinkedWithNintendoAccount() が true を返す
        @post
            - *pOut が連携状態を更新する手続きを実施可能

        @details
            ネットワークサービスアカウントとニンテンドーアカウントを連携状態を更新する手続きを再開します。
            再開された手続きにおいて OAuthProcedure::GetRequest() を使用することはできません。
    */
    Result ResumeProcedureToUpdateLinkageStateOfNintendoAccount(NintendoAccountLinkageStateUpdateProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントとNNIDを連携する手続きのためのオブジェクトを取得します。

        @param[out] pOut        手続きを実施するためのオブジェクト

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理が成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOut が手続きを実施可能

        @details
            ニンテンドーアカウントとNNIDを連携する手続きのためのオブジェクトを取得します。
            この手続きでは、認証や認可の結果は NintendoAccountNnidLinkageProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
            従って NintendoAccountNnidLinkageProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。
    */
    Result CreateProcedureToLinkNnidWithNintendoAccount(NintendoAccountNnidLinkageProcedure* pOut) NN_NOEXCEPT;

    /**
        @brief ニンテンドーアカウントとNNIDを連携する手続きのためのオブジェクトを再開します。

        @param[out] pOut        手続きを実施するためのオブジェクト
        @param[in] sessionId    OAuthProcedure::Suspend() によって取得したセッション識別子

        @retresult
            @handleresult{
                nn::ResultSuccess,
                再開に成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOut が手続きを実施可能

        @details
            ニンテンドーアカウントとNNIDを連携する手続きを再開します。
            再開された手続きにおいて OAuthProcedure::GetRequest() を使用することはできません。
    */
    Result ResumeProcedureToLinkNnidWithNintendoAccount(NintendoAccountNnidLinkageProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

    /**
        @brief アプリケーションに対してニンテンドーアカウントに関するユーザーの認可を提供するための手続きオブジェクトを取得します。

        @param[out] pOut 手続きを実施するためのオブジェクト
        @param[in] sessionId 手続きオブジェクトを指すセッションID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOut が手続きを実施可能

        @details
            アプリケーションに対してニンテンドーアカウントに関するユーザーの認可を提供するための手続きオブジェクトを取得します。
            この手続きでは、認証や認可の結果は NintendoAccountApplicationAuthorizationProcedure::GetRequest() で得られるリクエストパラメータに対するコールバックURIとして返却されます。
            従って NintendoAccountApplicationAuthorizationProcedure::ApplyResponseAsync() には、リダイレクトされたLocationヘッダの内容をコールバックURIを含めて指定してください。
    */
    Result ProxyProcedureToAcquireApplicationAuthorizationForNintendoAccount(NintendoAccountApplicationAuthorizationProcedure* pOut, const SessionId& sessionId) NN_NOEXCEPT;

    /**
        @brief 指定した ApplicationId のコンテキストでキャッシュされている、必要なネットワークサービスの資格情報を取得します。

        @param[out] pOut 必要なネットワークサービスの資格情報
        @param[in] applicationId 対象のアプリケーションのID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - 初期化状態
            - pOut != nullptr
            - IsNetworkServiceAccountAvailable() が true を返す
        @post
            - *pOut が NetworkServiceLicense のいずれかの値を保持

        @details
            指定した ApplicationId において、このユーザーがネットワークサービスを利用するために必要な資格情報を、現時点で判明している範囲で取得します。
    */
    Result GetRequiredLicenseCache(NetworkServiceLicense* pOut, const ApplicationId& applicationId) const NN_NOEXCEPT;

    /**
        @brief 指定した ApplicationId のコンテキストでキャッシュされている、必要なネットワークサービスの資格情報を破棄します。

        @param[in] applicationId 対象のアプリケーションのID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
        @endretresult

        @pre
            - *this が有効
        @post
            - GetRequiredLicenseCache() が NetworkServiceLicense_None を返す。

        @details
            GetRequiredLicenseCache() が NetworkServiceLicense_None を返すようキャッシュを破棄します。
    */
    Result InvalidateRequiredLicenseCache(const ApplicationId& applicationId) NN_NOEXCEPT;

    /**
        @brief 指定した ApplicationId のコンテキストでキャッシュされている、ネットワークサービスアカウントのIDトークンを破棄します。

        @param[in] applicationId 対象のアプリケーションのID

        @retresult
            @handleresult{
                nn::ResultSuccess,
                処理に成功しました。
            }
            @handleresult{
                nn::account::ResultNetworkServiceAccountUnavailable,
                ネットワークサービスアカウントが利用可能ではありません。
                EnsureNetworkServiceAccountAvailable() を実行することでネットワークサービスアカウントを利用可能な状態にし、このエラーを解消することができます。
            }
        @endretresult

        @pre
            - *this が有効
            - IsNetworkServiceAccountAvailable() が true を返す

        @details
            指定した ApplicationId のコンテキストでキャッシュされている、ネットワークサービスアカウントのIDトークンを破棄します。
    */
    Result InvalidateNetworkServiceAccountIdTokenCache(const ApplicationId& applicationId) NN_NOEXCEPT;

    // 非公開関数
    Result SynchronizeProfileAsyncIfTimeElapsed(bool* pOutMatched, AsyncContext* pOutContext, TimeSpan span) NN_NOEXCEPT;
    Result UnregisterAsync(AsyncContext* pOutContext) NN_NOEXCEPT;
    Result TryRecoverNintendoAccountUserStateAsync(AsyncContext* pOutContext) NN_NOEXCEPT;

    // デバッグ用内部向け関数
    Result DebugSetNetworkServiceAccountAvailabilityError(Result expect) NN_NOEXCEPT;
    Result DebugUnlinkNintendoAccountAsync(AsyncContext* pOutContext) NN_NOEXCEPT;
};

/**
    @brief アカウントのバックグラウンド処理を一時停止します。
    @details
        アカウントのバックグラウンド処理を一時停止します。
        システム全体を通して、本 API で取得した有効な Declaration オブジェクトが存在する間、バックグラウンド処理は実行されません。

        システムプロセスから本関数を呼ぶとデッドロックする場合があります。
*/
Declaration SuspendBackgroundActivity() NN_NOEXCEPT;

}} // ~namespace nn::account
