﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using System.Runtime.InteropServices;

namespace Nintendo.ServiceFramework.Validation
{
    using Localization;

    public class StructValidator
    {
        /// <summary>
        /// 与えられた型が SF で使用する構造体として正しいかどうかを validate する
        /// </summary>
        /// <param name="t">validate する対象の型</param>
        public void Validate(Type t)
        {
            if (t.IsAutoLayout)
            {
                throw new InvalidStructException(@"SF-ST-InvalidAutoLayout", t, @"構造体型に指定する {0} 属性の引数に {1} を指定することはできません。".Ja(),
                    typeof(StructLayoutAttribute).FullName,
                    LayoutKind.Auto);
            }
            var fields = t.GetFields(BindingFlags.Instance | BindingFlags.NonPublic | BindingFlags.Public);
            var externalStructAttribute = ExternalStructAttribute.GetSizeAndAlignment(t);
            if (externalStructAttribute == null)
            {
                foreach (var field in fields)
                {
                    ValidateField(field);
                }
            }
            else
            {
                if (!(fields.Length == 0))
                {
                    throw new InvalidStructException(@"SF-ST-InvalidExternalStruct:HasFields", t, @"{0} 属性で修飾された構造体型が、具体的なメンバフィールドを持っていますが、持つことはできません。".Ja(),
                        typeof(ExternalStructAttribute).FullName);
                }
            }
        }

        private static void ValidateFieldType(FieldInfo field, Type type)
        {
            if (SfStruct.CanAccept(type))
            {
                new StructValidator().Validate(type);
            }
            else if (SfBuiltIn.CanAccept(type))
            {
                // nop
            }
            else if (type.IsArray)
            {
                throw new InvalidFieldException(@"SF-ST-NotSupported:MultiRankArray", field, @"構造体フィールドに多次元配列が使用されています。使用できません。".Ja());
            }
            else
            {
                throw new InvalidFieldException(@"SF-ST-NotSupported:FieldType", field, @"構造体フィールドの型として使用できない型 {0} が使用されています。".Ja(), field.FieldType.FullName);
            }
        }

        private static void ValidateField(FieldInfo field)
        {
            if (!field.IsPublic)
            {
                throw new InvalidFieldException(@"SF-ST-NotPublicField", field, @"構造体フィールドは public である必要があります。".Ja());
            }
            if (field.FieldType.IsArray)
            {
                var length = FixedArrayAttribute.GetLength(field);
                if (length == null)
                {
                    throw new InvalidFieldException(@"SF-ST-NotSupported:DynamicArray", field, @"構造体フィールドに可変長配列が使用されています。{0} を使用して固定長を指定してください。".Ja(), typeof(FixedArrayAttribute).FullName);
                }
                ValidateFieldType(field, field.FieldType.GetElementType());
            }
            else
            {
                if (FixedArrayAttribute.GetLength(field) != null)
                {
                    throw new InvalidFieldException(@"SF-ST-InvalidField:FixedArrayedScalar", field, @"配列でない構造体フィールドが {0} で修飾されています。""[]"" を付け忘れている可能性があります。".Ja(), typeof(FixedArrayAttribute).FullName);
                }
                ValidateFieldType(field, field.FieldType);
            }
        }
    }
}
