﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;
using System.Runtime.InteropServices;

namespace Nintendo.ServiceFramework
{
    public class SfStruct : SfValueType
    {
        private int m_Size;
        private int m_Alignment;

        public SfStruct(Type target)
            : base(target)
        {
            new Validation.StructValidator().Validate(target);
            var sizeAndAlignment = GetSizeAndAlignment();
            this.m_Size = sizeAndAlignment.Size;
            this.m_Alignment = sizeAndAlignment.Alignment;
        }

        private SizeAndAlignment GetSizeAndAlignmentForField(FieldInfo field)
        {
            var type = field.FieldType;
            var length = 1;
            if (type.IsArray)
            {
                type = type.GetElementType();
                length = FixedArrayAttribute.GetLength(field).Value;
            }
            var sfStructBase = (SfValueType)type.GetSfEntity();
            return new SizeAndAlignment
            {
                Size = sfStructBase.Size * length,
                Alignment = sfStructBase.Alignment,
            };
        }

        private SizeAndAlignment GetSizeAndAlignment()
        {
            var externalStructAttribute = ExternalStructAttribute.GetSizeAndAlignment(InnerType);
            if (externalStructAttribute != null)
            {
                return externalStructAttribute;
            }
            else
            {
                return BinaryFormatUtility.GetSizeAndAlignment(GetFields().Select(GetSizeAndAlignmentForField));
            }
        }

        public static bool CanAccept(Type t)
        {
            // struct のみを対象とする
            return true
                && !SfBuiltIn.CanAccept(t)
                && t.IsValueType
                && !t.IsEnum
                && !t.IsPrimitive
                && t != typeof(void);
        }

        internal override IEnumerable<SfEntity> GetReferredEntities()
        {
            return GetStronglyReferredEntities();
        }

        public IEnumerable<FieldInfo> GetFields()
        {
            return InnerType.GetFields(BindingFlags.Instance | BindingFlags.Public);
        }

        internal override IEnumerable<SfEntity> GetStronglyReferredEntities()
        {
            return from field in GetFields()
                   let type = field.FieldType
                   let eType = type.IsArray ? type.GetElementType() : type
                   select eType.GetSfEntity();
        }

        public override int Size
        {
            get
            {
                return m_Size;
            }
        }

        public override int Alignment
        {
            get
            {
                return m_Alignment;
            }
        }

        internal bool IsExternal()
        {
            return ExternalStructAttribute.GetSizeAndAlignment(InnerType) != null;
        }
    }
}
