﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Nintendo.ServiceFramework
{
    using Localization;

    /// <summary>
    /// Type から SfEntity を取得するためのクラス
    /// </summary>
    public static class SfEntityGetter
    {
        #region GetEntity

        private static Dictionary<Type, SfEntity> s_SfEntityMap = new Dictionary<Type, SfEntity>();

        /// <summary>
        /// 指定した型に対する適切なエンティティオブジェクトを取得する
        /// </summary>
        /// <param name="target">エンティティとする型</param>
        /// <returns>エンティティオブジェクト</returns>
        public static SfEntity GetSfEntity(this Type target)
        {
            // 同じ Type に対しては、必ず同じ SfEntity を返すために、結果を s_SfEntityMap にメモ化する
            lock (s_SfEntityMap)
            {
                var ret = default(SfEntity);
                if (!s_SfEntityMap.TryGetValue(target, out ret))
                {
                    ret = MakeSfEntity(target);
                    s_SfEntityMap.Add(target, ret);
                }
                return ret;
            }
        }

        private static readonly Type[] s_KnownSfEntityClassTypes = new Type[]
        {
            typeof(SfBuiltIn),
            typeof(SfInterface),
            typeof(SfStruct),
            typeof(SfNnResult),
            typeof(SfNnBuffer),
            typeof(SfNativeHandle),
        };

        private static SfEntity MakeSfEntity(Type target)
        {
            // s_KnownSfEntityClassTypes に登録されている各クラスに対して
            foreach (var entityType in s_KnownSfEntityClassTypes)
            {
                // CanAccept static メソッドを呼んで true が返ってきたら
                if ((bool)entityType.InvokeMember(@"CanAccept", BindingFlags.InvokeMethod | BindingFlags.Static | BindingFlags.Public, null, null, new object[] { target }))
                {
                    // そのクラスインスタンスを作成して返す
                    return (SfEntity)entityType.GetConstructor(new[] { typeof(Type) }).Invoke(new object[] { target });
                }
            }
            return HandleInvalidEntity(target);
        }

        private static SfEntity HandleInvalidEntity(Type target)
        {
            if (target == typeof(char))
            {
                throw new UnknownEntityException(@"SF-InvalidEntity:char", target, @"System.Char は使用できません。かわりに char_t を使用してください。".Ja());
            }
            throw new UnknownEntityException(@"SF-UnknownEntity", target, @"{0} で表されるエンティティが不明です。".Ja(), target);
        }

        #endregion
    }
}
