﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.Localization
{
    public enum FriendlyStringLanguage
    {
        Japanese,
        English,
    }

    /// <summary>
    /// ユーザが読みやすい文字列を表すための型
    /// </summary>
    public class FriendlyString
    {
        public string OriginalText { get; private set; }
        public FriendlyStringLanguage Language { get; private set; }

        public FriendlyString(string s, FriendlyStringLanguage language)
        {
            this.OriginalText = s;
            this.Language = language;
        }

        /// <summary>
        /// 環境で設定されている言語に合わせた文字列を返す(現実装では固定した文字列を返すのみ)
        /// </summary>
        /// <returns>環境で設定されている言語に合わせた文字列</returns>
        public override string ToString()
        {
            return OriginalText;
        }

        public static implicit operator string(FriendlyString x)
        {
            return x.ToString();
        }
    }

    public static class FriendlyStringExtension
    {
        /// <summary>
        /// 日本語文字列から FriendlyString のインスタンスを作成する
        /// </summary>
        /// <param name="s">日本語文字列</param>
        /// <returns>生成された FriendlyString インスタンス</returns>
        public static FriendlyString Ja(this string s)
        {
            return new FriendlyString(s, FriendlyStringLanguage.Japanese);
        }

        /// <summary>
        /// 英語文字列から FriendlyString のインスタンスを作成する
        /// </summary>
        /// <param name="s">英語文字列</param>
        /// <returns>生成された FriendlyString インスタンス</returns>
        public static FriendlyString En(this string s)
        {
            return new FriendlyString(s, FriendlyStringLanguage.English);
        }
    }
}
