﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Threading.Tasks;

namespace Nintendo.ServiceFramework.CppCode
{
    public class NnBufferCppCodeEmitter : EntityCppCodeEmitterBase<SfNnBuffer>
    {
        internal NnBufferCppCodeEmitter(SfNnBuffer entity)
            : base(entity)
        {
        }

        private static readonly string[] CommonIncludes = new[]
        {
            @"<nn/sf/sf_Types.h>",
        };

        public override string GetParameterString(string variableName, InOutType inOutType, bool friendly)
        {
            variableName = variableName == null ? string.Empty : " " + variableName;
            return string.Format(@"const {0}&{1}", GetCppTypeString(friendly), variableName);
        }

        public override string GetCppTypeString(bool friendly)
        {
            var prefix = friendly ? string.Empty : @"::";
            var t = Entity.InnerType;
            if (t == typeof(nn.sf.InBuffer))
            {
                return prefix + @"nn::sf::InBuffer";
            }
            if (t == typeof(nn.sf.OutBuffer))
            {
                return prefix + @"nn::sf::OutBuffer";
            }
            if (t.IsGenericType)
            {
                var gt = t.GetGenericTypeDefinition();
                if (gt == typeof(nn.sf.InArray<>))
                {
                    var et = t.GetGenericArguments()[0];
                    return prefix + string.Format(@"nn::sf::InArray<{0}>", et.GetSfEntity().GetCppTypeString(friendly));
                }
                if (gt == typeof(nn.sf.OutArray<>))
                {
                    var et = t.GetGenericArguments()[0];
                    return prefix + string.Format(@"nn::sf::OutArray<{0}>", et.GetSfEntity().GetCppTypeString(friendly));
                }
            }
            throw new WrongImplementationException();
        }

        public override IEnumerable<string> GetExternalIncludes()
        {
            return CommonIncludes;
        }

        public override void EmitForwardDeclarationCode(RawCppCodeGenerator rawGen)
        {
            // nop
        }

        public override void EmitDefinitionCode(RawCppCodeGenerator rawGen)
        {
            // nop
        }
    }

    public static class NnBufferCppCodeEmitterHelper
    {
        private static IEnumerable<string> GetBufferTransferInternalDirectionStrings(ParameterInfo p)
        {
            switch (BufferParameterAttribute.GetBufferTransferInternalDirection(p))
            {
                case BufferTransferInternalDirection.In:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_IN";
                        yield break;
                    }
                case BufferTransferInternalDirection.Out:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_OUT";
                        yield break;
                    }
            }
        }

        private static IEnumerable<string> GetBufferTransferModeStrings(ParameterInfo p)
        {
            var hipcMode = Hipc.BufferTransferModeAttribute.GetBufferTransferMode(p);
            switch (hipcMode.Value)
            {
                case Hipc.BufferTransferMode.MapAlias:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_ALIAS";
                        yield break;
                    }
                case Hipc.BufferTransferMode.PointerCopy:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_POINTER";
                        yield break;
                    }
                case Hipc.BufferTransferMode.AutoSelect:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_AUTO_SELECT";
                        yield break;
                    }
            }
        }

        private static IEnumerable<string> GetMapTransferSecurityStrings(ParameterInfo p)
        {
            var security = Hipc.MapTransferSecurityAttribute.GetMapTransferSecurity(p);
            switch (security)
            {
                case Hipc.MapTransferSecurity.Secure:
                    {
                        // 出力なし
                        yield break;
                    }
                case Hipc.MapTransferSecurity.NonSecure:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_SECURE";
                        yield break;
                    }
                case Hipc.MapTransferSecurity.NonDevice:
                    {
                        yield return "NN_SF_CMIF_DETAIL_BUFFER_ATTRIBUTE_HIPC_MAP_NON_DEVICE";
                        yield break;
                    }
            }
        }

        public static string GetBufferAttributeString(this ParameterInfo p)
        {
            var list = GetBufferTransferInternalDirectionStrings(p)
                .Concat(GetBufferTransferModeStrings(p))
                .Concat(GetMapTransferSecurityStrings(p));
            return string.Join(" | ", list.ToArray());
        }
    }
}
