﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Reflection;

namespace Nintendo.ServiceFramework.CppCode
{
    /// <summary>
    /// 対象のエンティティの C++ 上での完全修飾名を指定するための属性
    /// </summary>
    /// <remarks>この属性が指定されなかった場合には、.net 上の完全修飾名を元に自動的に付加される。</remarks>
    [Serializable]
    [AttributeUsage(AttributeTargets.Interface | AttributeTargets.Struct | AttributeTargets.Enum | AttributeTargets.Class, AllowMultiple = false)]
    public class CppFullNameAttribute : Attribute
    {
        /// <summary>
        /// C++ 上での完全修飾名
        /// </summary>
        private string FullName { get; set; }

        /// <summary>
        /// C++ 上での完全修飾名を指定してインスタンスを生成するコンストラクタ
        /// </summary>
        /// <param name="fullName">C++ 上での完全修飾名</param>
        public CppFullNameAttribute(string fullName)
        {
            this.FullName = fullName;
        }

        /// <summary>
        /// (内部用) 型から、CppFullName 属性の値を取得するためのユーティリティ
        /// </summary>
        /// <param name="t">対象の型</param>
        /// <returns>C++ 上での完全修飾名</returns>
        internal static string GetCppFullName(Type t)
        {
            var a = t.GetCustomAttribute<CppFullNameAttribute>();
            return a == null ? t.FullName.Replace(".", "::").Replace("+", "::") : a.FullName;
        }
    }

    /// <summary>
    /// 対象のエンティティを C++ 上で参照するために必要な #include のパスを指定するための属性
    /// </summary>
    [Serializable]
    [AttributeUsage(AttributeTargets.Interface | AttributeTargets.Struct | AttributeTargets.Enum, AllowMultiple = true)]
    public class CppRefPathAttribute : Attribute
    {
        /// <summary>
        /// 対象のエンティティを C++ 上で参照するために必要な #include のパス
        /// </summary>
        private string Path { get; set; }

        /// <summary>
        /// パスを指定してインスタンスを生成するコンストラクタ
        /// </summary>
        /// <param name="path">対象のエンティティを C++ 上で参照するために必要な #include のパス</param>
        public CppRefPathAttribute(string path)
        {
            this.Path = path;
        }

        /// <summary>
        /// (内部用) 型から、CppRefPath 属性の値を取得するためのユーティリティ
        /// </summary>
        /// <param name="t">対象の型</param>
        /// <returns>CppRefPath 値</returns>
        internal static IEnumerable<string> GetCppRefPaths(Type t)
        {
            return from a in t.GetCustomAttributes<CppRefPathAttribute>()
                   select a.Path;
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter, AllowMultiple = true)]
    public abstract class PreBaseAttribute : Attribute
    {
        public abstract string MakeCppCheckString(string name);

        public virtual string MakeAssertionFailureString(string name)
        {
            return MakeCppCheckString(name);
        }

        public virtual bool UsesProcessId
        {
            get
            {
                return false;
            }
        }

        public virtual bool IsSecurityCheck
        {
            get
            {
                return false;
            }
        }

        private static IEnumerable<PreBaseAttribute> GetParameterPreAttributesImpl(ParameterInfo p)
        {
            foreach (var e in p.GetCustomAttributes<PreBaseAttribute>())
            {
                yield return e;
            }
            foreach (var e in p.ParameterType.GetSfParameterElementType().GetCustomAttributes<PreBaseAttribute>())
            {
                yield return e;
            }
        }

        internal static IEnumerable<PreBaseAttribute> GetPreAttributes(ParameterInfo p, bool withSecureityCheck)
        {
            var x = GetParameterPreAttributesImpl(p);
            if (withSecureityCheck)
            {
                return x;
            }
            else
            {
                return x.Where(e => !e.IsSecurityCheck);
            }
        }
    }

    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter | AttributeTargets.Struct, AllowMultiple = true)]
    public class PreAttribute : PreBaseAttribute
    {
        public PreAttribute(string checkString)
        {
            this.m_CheckString = checkString;
        }

        private string m_CheckString;

        public override string MakeCppCheckString(string name)
        {
            return string.Format(@"NN_SF_DETAIL_APPLY_LAMBDA(_, ({0}), {1})", m_CheckString, name);
        }

        public override string MakeAssertionFailureString(string name)
        {
            return string.Format(@"{0} (where _ = {1})", m_CheckString, name);
        }
    }
}

namespace Nintendo.ServiceFramework.Applet
{
    [Serializable]
    [AttributeUsage(AttributeTargets.Parameter | AttributeTargets.Struct, AllowMultiple = true)]
    public class CheckAppletResourceUserIdAttribute : CppCode.PreBaseAttribute
    {
        public CheckAppletResourceUserIdAttribute()
        {
        }

        public override bool UsesProcessId
        {
            get
            {
                return true;
            }
        }

        public override bool IsSecurityCheck
        {
            get
            {
                return true;
            }
        }

        public override string MakeCppCheckString(string name)
        {
            return string.Format(@"NN_SF_CMIF_SERVER_DETAIL_DEFINE_DISPATCH_METHOD_APPLY_PROCESS_ID(_, CheckAppletResourceUserId(_processId, _), {0})", name);
        }
    }
}
