﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Threading;
using Nintendo.ControlTarget;
using Nintendo.InitializeSdev;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.SystemUpdateEdev
{
    partial class EdevSystemUpdate
    {
        private const int RunOnTargetInterval = 4;
        private const int ResetInterval = 20;
        private const int DevKitUpdateTimeoutSecond = 120;

        private string GetVersionString()
        {
            if (Argument.targetFirmwareVersion == null)
            {
                return string.Empty;
            }
            else
            {
                return Argument.targetFirmwareVersion;
            }
        }

        private void LogWarningIfProdFirmware()
        {
            string versionString = GetVersionString();

            bool isProd = FirmwareResourceAccessor.CheckProdVersion(versionString);
            bool isProdWithLog = FirmwareResourceAccessor.CheckProdWithLogVersion(versionString);
            if(isProd || isProdWithLog)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "To use \"{0}\" type firmware, you must start devkit in safe mode manually.",
                                                isProd ? "+ Prod" : "+ ProdWithLog");
            }
        }

        private ExitStatus UsbAccessCheck()
        {
            using (UsbDeviceAccessor usbAccess = new UsbDeviceAccessor())
            {
                int deviceNum = usbAccess.GetDeviceNum();
                if (deviceNum == 0)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "EDEV is not connected. Please check USB connection and EDEV is powered on.");
                    LogWarningIfProdFirmware();
                    return ExitStatus.Failure;
                }
                else if (deviceNum < 0)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_ERROR, "NX devkits enumeration via USB failed.");
                    return ExitStatus.Failure;
                }
            }
            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManager()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager.");
            if (ProcessAccessor.StopTargetManager() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");
            return ExitStatus.Success;
        }

        private ExitStatus StopTargetManagerForLast()
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Stop NintendoTargetManager .");
            if (ProcessAccessor.StopTargetManagerForLast() != Nintendo.InitializeSdev.ExitStatus.Success)
            {
                return ExitStatus.Failure;
            }
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "NintendoTargetManager Stopped.");

            return ExitStatus.Success;
        }

        private ExitStatus ValidateEdevVersion(FirmwareResourceSpecifier firmwareResource)
        {
            // Prod もしくは ProdWithLog の場合はK5鍵の機種のみ利用可能
            if (!firmwareResource.ValidateSetVersion())
            {
                return ExitStatus.Failure;
            }

            return ExitStatus.Success;
        }

        private ExitStatus GetEdevVersionInNormalMode(out InitializeEdev.TargetSpecifier.EdevVersion edevVersion, TargetInnerClass targetKey, string targetManagerPath, ref ProcessingProgress progress)
        {
            ExitStatus eStatus;
            string devkitHardwareType;

            // LOG.LogLine("The safe mode may be preparing, so try to get configuration on normal mode.");

            eStatus = InitializeEdev.TargetManagerAccess.RegisterTarget(targetManagerPath, targetKey) == InitializeEdev.ExitStatus.Success ? ExitStatus.Success : ExitStatus.Failure;
            if (eStatus != ExitStatus.Success)
            {
                edevVersion = InitializeEdev.TargetSpecifier.EdevVersion.EDEV_ERROR;
                return eStatus;
            }
            eStatus = InitializeEdev.TargetManagerAccess.GetDevkitHardwareType(targetManagerPath, targetKey, out devkitHardwareType) == InitializeEdev.ExitStatus.Success ? ExitStatus.Success : ExitStatus.Failure;
            if (eStatus != ExitStatus.Success)
            {
                edevVersion = InitializeEdev.TargetSpecifier.EdevVersion.EDEV_ERROR;
                return eStatus;
            }
            if (devkitHardwareType == "EDEV")
            {
                // "EDEV" とは EDEV MP のことである
                edevVersion = InitializeEdev.TargetSpecifier.EdevVersion.EDEV_MP;
                firmwareResource.SetSdevVersion(InitializeSdev.TargetSpecifier.SdevVersion.EDEV_MP);
                return ExitStatus.Success;
            }

            edevVersion = InitializeEdev.TargetSpecifier.EdevVersion.EDEV_ERROR;
            return ExitStatus.Failure;
        }

        private ExitStatus DoDevKitUpdate(string devkitUpdaterPath, TargetInnerClass targetKey, string targetManagerPath)
        {
            LOG.LogLine("Do update.");

            // call RunOnTarget
            List<string> args1 = new List<string>() {"--target", GetTargetStringForRunOnTarget(targetKey.Value), "--reset",
                                "--failure-timeout", DevKitUpdateTimeoutSecond.ToString(),
                                "--pattern-not-found-failure", "Update Succeeded\\.", "--pattern-failure-exit", "Update Failed\\.",
                                "--extra-targetmanager-dir", targetManagerPath};
            List<string> args2 = new List<string>() { devkitUpdaterPath };
            List<string> args3 = new List<string>() { "--", "--spmariobootpc" };

            if (Argument.Verbose)
            {
                args1.Add("--verbose");
            }
            List<string> argsList = new List<string>();
            argsList.AddRange(args1);
            argsList.AddRange(args2);

            ProcessAccessor.DumpProcessArgumentList("RunOnTarget", argsList);

            // not display
            argsList.AddRange(args3);

            int retVar = Nintendo.RunOnTarget.CommandInterface.Main(argsList.ToArray());

            // MEMO: put dummy line feed for RunOnTarget log
            Console.WriteLine(string.Empty);

#if false
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  Wait 10 sec.");
            Thread.Sleep(10000);
#endif

            return retVar == 0 ? ExitStatus.Success : ExitStatus.Failure;
        }

        private string GetTargetStringForRunOnTarget(string targetKeyString)
        {
            if (targetKeyString.ToUpper() != "USB")
            {
                SerialNumber serial = new SerialNumber(targetKeyString);
                if (serial.GetString() == null)
                {
                    LOG.LogLine(LOG_LEVEL.LOG_WARN, "GetTargetStringForRunOnTarget: The serial number is invalid. ({0})", targetKeyString);
                    return "";
                }
                return serial.GetStringWithCheckDigit();
            }
            else
            {
                return "usb";
            }
        }

#if false
        private ExitStatus ExecutePluginsBeforeInitialize()
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePre, false);
            return bRet ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus ExecutePluginsAfterInitialize(TargetInnerClass targetKey, TargetSpecifier.EdevVersion edevVersion)
        {
            bool bRet = Plugins.Execute(Plugins.ExecuteTimingType.ExecutePost, false, targetKey, TargetSpecifier.GetDevkitVersion(edevVersion));
            return bRet ? ExitStatus.Success : ExitStatus.Failure;
        }
#endif

        private ExitStatus RestartTarget(string targetKey, string targetManagerPath)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reset target.");
            using (var tmapiAccessor = new TargetManagerAccessor())
            {
                tmapiAccessor.SetExtraTargetManagerDirectory(targetManagerPath);
                tmapiAccessor.EnsureStart();

                using (var target_ = tmapiAccessor.GetTarget(tmapiAccessor.FindTarget(targetKey)))
                {
                    tmapiAccessor.RebootTarget(target_);
                }

            }

            Thread.Sleep(10000);

            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "Reset target finished.");

            return ExitStatus.Success;
        }

        private ExitStatus ActivateTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return InitializeEdev.TargetManagerAccess.ActivateTarget(targetManagerPath, targetKey) == InitializeEdev.ExitStatus.Success ? ExitStatus.Success : ExitStatus.Failure;
        }

        private ExitStatus DisconnectTarget(TargetInnerClass targetKey, string targetManagerPath)
        {
            return InitializeEdev.TargetManagerAccess.DisconnectTarget(targetManagerPath, targetKey) == InitializeEdev.ExitStatus.Success ? ExitStatus.Success : ExitStatus.Failure;
        }
    }
}
