﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Threading.Tasks.Dataflow;
using Nintendo.Foundation.IO;
using System.IO.Ports;
using System.Globalization;
using Nintendo.ControlTarget;
using Nintendo.RunOnTarget;
using System.Diagnostics;

namespace ReadHostBridgeLog
{
    public class ReadHostBridgeLogArgument
    {
        [CommandLineOption("address",
            Description = "Set set address.", IsRequired = true)]
        public string Address { get; set; }

        [CommandLineOption("success-timeout", DefaultValue = -1,
            Description = "Set timeout as success.")]
        public int SuccessTimeOut { get; set; }

        [CommandLineOption("failure-timeout", DefaultValue = -1,
            Description = "Set timeout as failure.")]
        public int FailureTimeOut { get; set; }

        [CommandLineOption("pattern-success-exit",
            Description = "Set exit pattern(regex) as success.")]
        public string[] SuccessExitPattern { get; set; }

        [CommandLineOption("pattern-failure-exit",
            Description = "Set exit pattern(regex) as failure.")]
        public string[] FailureExitPattern { get; set; }

        [CommandLineOption("pattern-reset-success-timeout",
            Description = "Set pattern(regex) to reset success timeout.")]
        public string[] ResetSuccessTimeOutPattern { get; set; }

        [CommandLineOption("pattern-reset-failure-timeout",
            Description = "Set pattern(regex) to reset failure timeout.")]
        public string[] ResetFailureTimeOutPattern { get; set; }

        [CommandLineOption("pattern-not-found-failure",
            Description = "Set pattern(regex) to add condition for failure when pattern is not found.")]
        public string[] NotFoundFailurePatterns { get; set; }

        [CommandLineOption('v', "verbose",
            Description = "Output trace logs.")]
        public bool Verbose { get; set; }

        private ExitCondition ExitCondition;

        public int Run()
        {
            using (var serialAccessor = new HostBridgeAccessor(NetworkUtility.Parse(this.Address, 10023)))
            {
                ExitCondition = new ExitCondition(
                    false,
                    SuccessExitPattern,
                    FailureExitPattern,
                    SuccessTimeOut,
                    FailureTimeOut,
                    ResetSuccessTimeOutPattern,
                    ResetFailureTimeOutPattern,
                    NotFoundFailurePatterns,
                    false);

                var broadCast = new BroadcastBlock<string>(s => s);

                using (var link1 = serialAccessor.ReadPort.LinkTo(broadCast))
                using (var link2 = broadCast.LinkTo(DataflowConsole.Instance.GetNamedTarget(this.Address)))
                {
                    var exitStatus = WaitingUtility.WaitExitCondition(broadCast, ExitCondition, new CancellationTokenSource(), null);

                    return exitStatus == Nintendo.RunOnTarget.ExitStatus.Success ? 0 : 1;
                }
            }
        }
    }

    class Program
    {
        static void Main(string[] args)
        {
            Thread.CurrentThread.CurrentUICulture = new CultureInfo("en", true);

            try
            {
                ReadHostBridgeLogArgument parsed;
                var parser = new Nintendo.Foundation.IO.CommandLineParser();

                if (false == parser.ParseArgs<ReadHostBridgeLogArgument>(args, out parsed))
                {
                    System.Environment.Exit(1);
                }

                DataflowConsole.Instance.SetOption(parsed.Verbose);
                ConsoleApplicationTraceListener.SetGlobal(parsed.Verbose);

                System.Environment.Exit(parsed.Run());
            }
            catch (Exception exception)
            {
                PrintException(exception);
                System.Environment.Exit(1);
            }
        }

        public static void PrintException(Exception exception)
        {
            Console.Error.WriteLine("[ERROR] {0}", exception.Message);
            Trace.WriteLine($"StackTrace: {exception.StackTrace}");
        }

    }
}
