﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;

using LOG = Nintendo.InitializeSdev.Logger;
using LOG_LEVEL = Nintendo.InitializeSdev.Logger.Level;

namespace Nintendo.InitializeSdev
{
    public class ProcessAccessor
    {
        public static ExitStatus StopTargetManager()
        {
            ExitStatus status = StopProcesses("NintendoTargetManager");
            if (status != ExitStatus.Success)
            {
                LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Failed to terminate NintendoTargetManager process. Please terminate it manually and retry.");
            }
            return status;
        }

        public static ExitStatus StopTargetManagerForLast()
        {
            return TaskKillCommand("NintendoTargetManager.exe");
        }

        private static ExitStatus StopProcesses(string processName)
        {
            ExitStatus status = ExitStatus.Success;
            Process[] ps = Process.GetProcessesByName(processName);
            int loopConunt = 0;
            while (ps.Length > 0 && loopConunt++ < 10)
            {
                // 1: try to close process
                Process p = ps[0];
                bool closeRet = p.CloseMainWindow();
                if (closeRet)
                {
                    p.WaitForExit(10000);
                    if (p.HasExited)
                    {
                        ps = Process.GetProcessesByName(processName);
                        continue;
                    }
                }

                // 2: try to terminate process
                p.Kill();
                p.WaitForExit(10000);
                ps = Process.GetProcessesByName(processName);
            }

            ps = Process.GetProcessesByName(processName);
            if (ps.Length > 0)
            {
                return ExitStatus.Failure;
            }

            return status;
        }

        private static ExitStatus TaskKillCommand(string processName)
        {
            string taskKillCommandPath = System.Environment.ExpandEnvironmentVariables(@"%SystemRoot%\System32\taskkill.exe");
            List<string> argList = new List<string>();
            argList.Add("/IM");
            argList.Add(processName);
            int iRet = DoProcess(taskKillCommandPath, argList, null);
            return ExitStatus.Success;
        }

        public static int DoProcess(string processPath, List<string> argumentList, DataReceivedEventHandler dataReceiveHandler, int timeout = 0)
        {
            string args = ConvertArgumentListToString(argumentList);

            int exitCode = -1;
            ProcessStartInfo psInfo = new ProcessStartInfo();
            // ProcessOutputBuffer = null;
            psInfo.FileName = processPath;
            psInfo.Arguments = args;
            psInfo.CreateNoWindow = true;
            psInfo.UseShellExecute = false;
            psInfo.RedirectStandardOutput = dataReceiveHandler != null ? true : false;
            psInfo.RedirectStandardError = dataReceiveHandler != null ? true : false;
            using (Process hProcess = Process.Start(psInfo))
            {
                if(dataReceiveHandler != null)
                {
                    hProcess.OutputDataReceived += dataReceiveHandler;
                    hProcess.ErrorDataReceived += dataReceiveHandler;
                    hProcess.BeginOutputReadLine();
                    hProcess.BeginErrorReadLine();
                }
                if (timeout == 0)
                {
                    hProcess.WaitForExit();
                }
                else
                {
                    if(!hProcess.WaitForExit(timeout))
                    {
                        LOG.LogLine(LOG_LEVEL.LOG_ERROR, "Process execution is timeout. ({0})", processPath);
                        return -1;
                    }
                }
                exitCode = hProcess.ExitCode;
            }
            return exitCode;
        }

        public static string ConvertArgumentListToString(List<string> argumentList)
        {
            string args = null;
            foreach (string arg in argumentList)
            {
                args += arg + " ";
            }
            return args;
        }

        public static void DumpProcessArgumentList(string commandName, List<string> argumentList)
        {
            LOG.LogLine(LOG_LEVEL.LOG_INFO2, "  {0} argument :", commandName);
            foreach (string singleArg in argumentList)
            {
                LOG.LogLine(LOG_LEVEL.LOG_INFO2, "      {0}", singleArg);
            }
        }
    }
}
