﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ControlTargetManagerPlugin.h"

using namespace System::Text;

#include <msclr/marshal.h>
using namespace msclr::interop;

namespace Nintendo {
namespace ControlTarget {
namespace Tmapi {

TargetAccessor::TargetAccessor(nn::tm::TargetHandle targetHandle)
    : m_TargetHandle(gcnew TargetHandle(targetHandle))
{
}

TargetAccessor::TargetAccessor(TargetHandle^ targetHandle)
    : m_TargetHandle(targetHandle)
{
}

void TargetAccessor::EnsureConnect()
{
    if(!IsConnected())
    {
        Connect();
    }
}

void TargetAccessor::EnsureDisconnect()
{
    if(IsConnected())
    {
        Disconnect();
    }
}

bool TargetAccessor::IsConnected()
{
    bool isConnected = false;
    TMAPI_DO(nn::tm::GetTargetIsConnected(&isConnected, m_TargetHandle->GetHandle()));
    return isConnected;
}

bool TargetAccessor::IsSleeping()
{
    bool isSleeping = false;
    TMAPI_DO(nn::tm::GetTargetIsSleeping(&isSleeping, m_TargetHandle->GetHandle()));
    return isSleeping;
}

void TargetAccessor::PowerOff()
{
    TMAPI_DO( nn::tm::RemoveBattery(m_TargetHandle->GetHandle()) );
}

void TargetAccessor::PowerOn()
{
    TMAPI_DO(nn::tm::InsertBattery(m_TargetHandle->GetHandle()));
}

void TargetAccessor::Reset()
{
    TMAPI_DO(nn::tm::ResetTargetHard(m_TargetHandle->GetHandle()));
}

void TargetAccessor::Reboot()
{
    TMAPI_DO(nn::tm::ResetTargetHard(m_TargetHandle->GetHandle()));
}

void TargetAccessor::RebootHard()
{
    TMAPI_DO(nn::tm::ResetTargetHard(m_TargetHandle->GetHandle()));
}

void TargetAccessor::Connect()
{
    TMAPI_DO(nn::tm::ConnectTarget(m_TargetHandle->GetHandle()));
}

void TargetAccessor::Disconnect()
{
    TMAPI_DO(nn::tm::DisconnectTarget(m_TargetHandle->GetHandle()));
}

void TargetAccessor::SetDefault()
{
    TMAPI_DO(nn::tm::SetDefaultTarget(m_TargetHandle->GetHandle()));
}

void TargetAccessor::SetName(String^ name)
{
    pin_ptr<const wchar_t> wch = PtrToStringChars(name);

    size_t convertedChars = 0;
    size_t bufferSize = (name->Length + 1) * 2;
    char* buffer = nullptr;
    try
    {
        nn::tm::TargetName targetName = { 0 };
        errno_t err = wcstombs_s(&convertedChars, targetName.name, sizeof( targetName), wch, bufferSize);
        TMAPI_DO(nn::tm::SetTargetName(m_TargetHandle->GetHandle(), &targetName ));

        if(err != 0)
        {
            throw gcnew Exception("Failed to SetName");
        }
    }
    finally
    {
        if(buffer)
        {
            delete buffer;
        }
    }
}

String^ TargetAccessor::TakeScreenshot(String^ directory, String^ fileName)
{
    marshal_context ^ context = gcnew marshal_context();
    char outPath[256];
    TMAPI_DO(nn::tm::TakeScreenshot(outPath, sizeof(outPath), m_TargetHandle->GetHandle(), context->marshal_as<const char*>(directory), context->marshal_as<const char*>(fileName)));
    delete context;
    return gcnew String(outPath);
}

void TargetAccessor::CreateTargetProgramDump(String^ filePath, Tmapi::DumpType dumpType)
{
    marshal_context ^ context = gcnew marshal_context();

    TMAPI_DO(nn::tm::CreateTargetProgramDump(m_TargetHandle->GetHandle(), context->marshal_as<const char*>(filePath), static_cast<nn::tm::DumpType>(dumpType)));
    delete context;
}

void TargetAccessor::SetTargetWorkingDirectory(String^ workingDirectory)
{
    marshal_context ^ context = gcnew marshal_context();

    const char* mbcsBuffer = context->marshal_as<const char*>(workingDirectory);

    if (nn::tm::TargetWorkingDirectoryBufferLength <= strnlen_s(mbcsBuffer, nn::tm::TargetWorkingDirectoryBufferLength))
    {
        throw gcnew Exception("Too long target working directory.");
    }

    nn::tm::TargetWorkingDirectory tmWorkingDirectory;
    strncpy_s(
        tmWorkingDirectory.directory,
        mbcsBuffer,
        nn::tm::TargetWorkingDirectoryBufferLength);

    TMAPI_DO(nn::tm::SetTargetWorkingDirectory(m_TargetHandle->GetHandle(), &tmWorkingDirectory));
}

String^ TargetAccessor::GetTargetWorkingDirectory()
{
    nn::tm::TargetWorkingDirectory tmWorkingDirectory;
    TMAPI_DO(nn::tm::GetTargetWorkingDirectory(&tmWorkingDirectory, m_TargetHandle->GetHandle()));

    return gcnew String(tmWorkingDirectory.directory);
}

TargetInfo::TargetInfo(target_info *pTargetInfo)
{
    m_pTargetInfo = new target_info();
    *m_pTargetInfo = *pTargetInfo;
}

TargetInfo::TargetInfo(nn::tm::DiscoveryHandle discoveryHandle)
{
    target_info targetInfo;

    nn::tm::TargetName              TargetName = { 0 };
    TMAPI_DO(nn::tm::GetDiscoveryTargetName(&TargetName, discoveryHandle));
    strcpy_s(targetInfo.m_Name, sizeof(targetInfo.m_Name), TargetName.name);

    nn::tm::TargetIpAddress         TargetIpAddress = { 0 };
    TMAPI_DO(nn::tm::GetDiscoveryTargetIpAddress(&TargetIpAddress, discoveryHandle));
    strcpy_s(targetInfo.m_IPAddress, sizeof(targetInfo.m_IPAddress), TargetIpAddress.ipAddress);

    nn::tm::TargetMacAddress        TargetMacAddress = { 0 };
    TMAPI_DO(nn::tm::GetDiscoveryTargetMacAddress(&TargetMacAddress, discoveryHandle));
    strcpy_s(targetInfo.m_MacAddress, sizeof(targetInfo.m_MacAddress), TargetMacAddress.macAddress);

    nn::tm::TargetSerialNumber      TargetSerialNumber = { 0 };
    TMAPI_DO(nn::tm::GetDiscoveryTargetSerialNumber(&TargetSerialNumber, discoveryHandle));
    strcpy_s(targetInfo.m_SerialNumber, sizeof(targetInfo.m_SerialNumber), TargetSerialNumber.serialNumber);

    nn::tm::TargetHardwareType      TargetHardwareType = nn::tm::TargetHardwareType_Unknown;
    TMAPI_DO(nn::tm::GetDiscoveryTargetHardwareType(&TargetHardwareType, discoveryHandle));
    if (TargetHardwareType == nn::tm::TargetHardwareType_Sdev)
    {
        strcpy_s(targetInfo.m_HardwareType, sizeof(targetInfo.m_HardwareType), "SDEV");
    }
    else if (TargetHardwareType == nn::tm::TargetHardwareType_Edev)
    {
        strcpy_s(targetInfo.m_HardwareType, sizeof(targetInfo.m_HardwareType), "EDEV");
    }
    else if (TargetHardwareType == nn::tm::TargetHardwareType_Sim)
    {
        strcpy_s(targetInfo.m_HardwareType, sizeof(targetInfo.m_HardwareType), "Sim");
    }
    else
    {
        strcpy_s(targetInfo.m_HardwareType, sizeof(targetInfo.m_HardwareType), "Unknown");
    }

    nn::tm::TargetConnectionType    TargetConnectionType = nn::tm::TargetConnectionType_Unknown;
    TMAPI_DO(nn::tm::GetDiscoveryTargetConnectionType(&TargetConnectionType, discoveryHandle));
    if (TargetConnectionType == nn::tm::TargetConnectionType_Usb)
    {
        strcpy_s(targetInfo.m_ConnectionType, sizeof(targetInfo.m_ConnectionType), "USB");
    }
    else if (TargetConnectionType == nn::tm::TargetConnectionType_Ethernet)
    {
        strcpy_s(targetInfo.m_ConnectionType, sizeof(targetInfo.m_ConnectionType), "Ethernet");
    }
    else
    {
        strcpy_s(targetInfo.m_ConnectionType, sizeof(targetInfo.m_ConnectionType), "Unknown");
    }

    // Only attempt to acquire the agent version if the target is an EDEV.
    // Else, set the agent version to the default value.
    nn::tm::TargetAgentVersion TargetAgentVersion = { 0 };
    if (TargetHardwareType == nn::tm::TargetHardwareType_Edev)
    {
        TMAPI_DO(nn::tm::GetDiscoveryTargetAgentVersion( &TargetAgentVersion, discoveryHandle ));
    }
    strcpy_s(targetInfo.m_TMAVersion, sizeof( targetInfo.m_TMAVersion ), (const char*)System::Runtime::InteropServices::Marshal::StringToHGlobalAnsi( String::Format( "{0}.{1}.{2}.{3}", TargetAgentVersion.major, TargetAgentVersion.minor, TargetAgentVersion.micro, TargetAgentVersion.revision ) ).ToPointer());

    strcpy_s(targetInfo.m_ConnectStatus, sizeof(targetInfo.m_ConnectStatus), "Unknown");
    targetInfo.m_DiscoveryHandle = discoveryHandle;

    m_pTargetInfo = new target_info();
    *m_pTargetInfo = targetInfo;
}


TargetInfo::~TargetInfo()
{
    if (m_pTargetInfo)
    {
        delete m_pTargetInfo;
        m_pTargetInfo = nullptr;
    }
}

TargetInfo::!TargetInfo()
{
    if (m_pTargetInfo)
    {
        delete m_pTargetInfo;
        m_pTargetInfo = nullptr;
    }
}

target_info& TargetInfo::GetRawInfo()
{
    return *m_pTargetInfo;
}

String^ TargetInfo::GetIpAddress()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_IPAddress);
}

String^ TargetInfo::GetName()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_Name);
}

String^ TargetInfo::GetTmaVersion()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_TMAVersion);
}

String^ TargetInfo::GetMacAddress()
{
    return msclr::interop::marshal_as<String^>( m_pTargetInfo->m_MacAddress );
}
String^ TargetInfo::GetConnectionType()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_ConnectionType);
}

String^ TargetInfo::GetHardwareType()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_HardwareType);
}

String^ TargetInfo::GetSerialNumber()
{
    return msclr::interop::marshal_as<String^>(m_pTargetInfo->m_SerialNumber);
}

String^ TargetInfo::ToString()
{
    auto builder = gcnew StringBuilder();

    builder->AppendLine("TargetInfo:");
    builder->AppendLine(String::Format("    Name:           {0}", GetName()));
    builder->AppendLine(String::Format("    HardwareType:   {0}", GetHardwareType()));
    builder->AppendLine(String::Format("    ConnectionType: {0}", GetConnectionType()));
    builder->AppendLine(String::Format("    TmaVersion:     {0}", GetTmaVersion()));
    builder->AppendLine(String::Format("    SerialNumber:   {0}", GetSerialNumber()));
    builder->AppendLine(String::Format("    IpAddress:      {0}", GetIpAddress()));
    builder->AppendLine(String::Format("    MacAddress:     {0}", GetMacAddress()));

    return builder->ToString();
}

TargetHandle::TargetHandle(nn::tm::TargetHandle handle)
{
    mp_Handle = new nn::tm::TargetHandle();
    mp_Handle->handle = handle.handle;
}

TargetHandle::~TargetHandle()
{
    this->!TargetHandle();
}

TargetHandle::!TargetHandle()
{
    if (mp_Handle != nullptr)
    {
        delete mp_Handle;
        mp_Handle = nullptr;
    }
}

DiscoveryHandle::DiscoveryHandle(nn::tm::DiscoveryHandle handle)
{
    mp_Handle = new nn::tm::TargetHandle();
    mp_Handle->handle = handle.handle;
}


DiscoveryHandle::~DiscoveryHandle()
{
    this->!DiscoveryHandle();
}

DiscoveryHandle::!DiscoveryHandle()
{
    if (mp_Handle != nullptr)
    {
        delete mp_Handle;
        mp_Handle = nullptr;
    }
}

RegisteredTargetInfo::RegisteredTargetInfo(
    TargetHandle^ targetHandle,
    String^ name,
    String^ hardwareType,
    String^ connectionType,
    String^ tmaVersion,
    String^ serialNumber,
    String^ ipAddress,
    String^ macAddress,
    String^ status)
{
    m_TargetHandle = targetHandle;
    m_Name = name;
    m_HardwareType = hardwareType;
    m_ConnectionType = connectionType;
    m_TmaVersion = tmaVersion;
    m_SerialNumber = serialNumber;
    m_IpAddress = ipAddress;
    m_MacAddress = macAddress;
    m_Status = status;
}

TargetHandle^ RegisteredTargetInfo::GetTargetHandle()
{
    return m_TargetHandle;
}

String^ RegisteredTargetInfo::GetName()
{
    return m_Name;
}

String^ RegisteredTargetInfo::GetHardwareType()
{
    return m_HardwareType;
}

String^ RegisteredTargetInfo::GetConnectionType()
{
    return m_ConnectionType;
}

String^ RegisteredTargetInfo::GetTmaVersion()
{
    return m_TmaVersion;
}

String^ RegisteredTargetInfo::GetSerialNumber()
{
    return m_SerialNumber;
}

String^ RegisteredTargetInfo::GetIpAddress()
{
    return m_IpAddress;
}

String^ RegisteredTargetInfo::GetMacAddress()
{
    return m_MacAddress;
}


String^ RegisteredTargetInfo::GetStatus()
{
    return m_Status;
}

String^ RegisteredTargetInfo::ToString()
{
    auto builder = gcnew StringBuilder();

    builder->AppendLine("RegisteredTargetInfoTargetInfo:");
    builder->AppendLine(String::Format("    TargetHandle:   {0}", GetTargetHandle()->GetHandle().handle));
    builder->AppendLine(String::Format("    Name:           {0}", GetName()));
    builder->AppendLine(String::Format("    HardwareType:   {0}", GetHardwareType()));
    builder->AppendLine(String::Format("    ConnectionType: {0}", GetConnectionType()));
    builder->AppendLine(String::Format("    TmaVersion:     {0}", GetTmaVersion()));
    builder->AppendLine(String::Format("    SerialNumber:   {0}", GetSerialNumber()));
    builder->AppendLine(String::Format("    IpAddress:      {0}", GetIpAddress()));
    builder->AppendLine(String::Format("    MacAddress:     {0}", GetMacAddress()));

    return builder->ToString();
}

TargetManagerAccessor::TargetManagerAccessor()
{
}

int32_t TargetManagerAccessor::GetTargetCount()
{
    int32_t count = 0;
    TMAPI_DO(nn::tm::GetTargetCount(&count));
    return count;
}

bool TargetManagerAccessor::HasDefaultTarget()
{
    return GetDefaultTargetHandle()->GetHandle().handle != nn::tm::InvalidHandle;
}

TargetHandle^ TargetManagerAccessor::GetDefaultTargetHandle()
{
    nn::tm::TargetHandle targetHandle;

    nn::tm::Error error = nn::tm::GetDefaultTarget(&targetHandle);

    if ( error == nn::tm::Error_TargetUnavailable)
    {
        return gcnew TargetHandle({nn::tm::InvalidHandle});
    }

    TMAPI_DO(error);

    return gcnew TargetHandle(targetHandle);
}

RegisteredTargetInfo^ TargetManagerAccessor::GetDefaultTargetInfo()
{
    TargetHandle^ targetHandle = GetDefaultTargetHandle();
    for each(RegisteredTargetInfo^ targetInfo in this->ListTargets())
    {
        if( targetInfo->GetTargetHandle()->GetHandle().handle == targetHandle->GetHandle().handle)
        {
            return targetInfo;
        }
    }

    return nullptr;
}

TargetAccessor^ TargetManagerAccessor::GetDefaultTarget()
{
    return GetTarget(GetDefaultTargetHandle());
}

TargetAccessor^ TargetManagerAccessor::GetTarget(TargetHandle^ targetHandle)
{
    return gcnew TargetAccessor(targetHandle);
}

List<TargetInfo^>^ TargetManagerAccessor::DetectTargets(TimeSpan timeOut)
{
    nn::tm::Error error = nn::tm::StartDiscovery( static_cast<int32_t>(timeOut.TotalMilliseconds) );

    if( error == nn::tm::Error_TargetDiscoveryAlreadyStarted ||
        error == nn::tm::Error_Ok )
    {
        nn::tm::DiscoveryState discoveryState = nn::tm::DiscoveryState_Uninitialized;
        TimeSpan interval    = TimeSpan::FromMilliseconds(100);
        for(TimeSpan current = TimeSpan::FromMilliseconds(0);
            current < timeOut;
            current = current.Add(interval))
        {
            TMAPI_DO(nn::tm::GetDiscoveryState( &discoveryState ));

            if(discoveryState == nn::tm::DiscoveryState_Complete)
            {
                break;
            }

            Threading::Thread::Sleep(interval);
        }
    }
    else
    {
        TMAPI_DO(error);
    }

    TMAPI_DO(nn::tm::AbortDiscovery());

    List<TargetInfo^>^ ret = gcnew List<TargetInfo^>();

    int32_t numTargets = 0;
    TMAPI_DO(nn::tm::GetDiscoveryTargetCount(&numTargets));

    nn::tm::DiscoveryHandle* discoveryHandles = new nn::tm::DiscoveryHandle[numTargets];
    error = nn::tm::GetDiscoveryTargets(discoveryHandles, &numTargets, numTargets);

    if (error == nn::tm::Error_Ok)
    {
        for (int iTarget = 0; iTarget < numTargets; ++iTarget)
        {
            ret->Add(gcnew TargetInfo(discoveryHandles[iTarget]));
        }
    }

    delete[] discoveryHandles;
    TMAPI_DO(error);

    return ret;
}

List<TargetHandle^>^ TargetManagerAccessor::ListTargetHandles()
{
    List<TargetHandle^>^ ret = gcnew List<TargetHandle^>();

    int32_t numTargets = GetTargetCount();
    nn::tm::TargetHandle* targetHandles = new nn::tm::TargetHandle[numTargets];

    nn::tm::Error error = nn::tm::GetTargets(targetHandles, &numTargets, numTargets);

    for(int i = 0; i < numTargets && error == nn::tm::Error_Ok; i++)
    {
        try
        {
            ret->Add(gcnew TargetHandle(targetHandles[i]));
        }
        catch (Nintendo::ControlTarget::Tmapi::TmapiException ^te)
        {
            error = static_cast<nn::tm::Error>(te->GetTmapiError());
            break;
        }
    }

    delete[] targetHandles;
    TMAPI_DO(error);

    return ret;
}

List<RegisteredTargetInfo^>^ TargetManagerAccessor::ListTargets()
{
    List<RegisteredTargetInfo^>^ ret = gcnew List<RegisteredTargetInfo^>();

    int32_t numTargets = GetTargetCount();
    nn::tm::TargetHandle* targetHandles = new nn::tm::TargetHandle[numTargets];

    nn::tm::Error error = nn::tm::GetTargets(targetHandles, &numTargets, numTargets);

    if( error == nn::tm::Error_Ok )
    {
        for (int i = 0; i < numTargets && error == nn::tm::Error_Ok; i++)
        {
            ret->Add(gcnew RegisteredTargetInfo(
                gcnew TargetHandle(targetHandles[i]),
                GetTargetName(targetHandles[i]),
                GetTargetHardwareType(targetHandles[i]),
                GetTargetConnectionType(targetHandles[i]),
                GetTargetTMAVersion(targetHandles[i]),
                GetTargetSerialNumber(targetHandles[i]),
                GetTargetIPAddress(targetHandles[i]),
                GetTargetMacAddress(targetHandles[i]),
                GetTargetStatus(targetHandles[i])));
        }
    }

    delete[] targetHandles;
    TMAPI_DO(error);

    return ret;
}

void TargetManagerAccessor::RegisterTarget(TargetInfo^ info)
{
    nn::tm::TargetHandle targetHandle;
    TMAPI_DO(nn::tm::AddTarget(&targetHandle, info->GetRawInfo().m_DiscoveryHandle));
}

void TargetManagerAccessor::RegisterTarget(String^ ip)
{
    nn::tm::TargetHandle targetHandle;

    marshal_context ^ context = gcnew marshal_context();
    nn::tm::TargetIpAddress targetIpAddress = { 0 };
    const char* tempIpAddress = context->marshal_as<const char*>( ip );
    strcpy_s( targetIpAddress.ipAddress, sizeof( targetIpAddress.ipAddress ), tempIpAddress );
    TMAPI_DO( nn::tm::AddTarget( &targetHandle, &targetIpAddress ) );
    delete context;
}

void TargetManagerAccessor::UnregisterTarget(TargetHandle^ targetHandle)
{
    TMAPI_DO(nn::tm::RemoveTarget(targetHandle->GetHandle()));
}

void TargetManagerAccessor::UnregisterAllTargets()
{
    auto targetHandles = ListTargetHandles();

    for each (TargetHandle^ targetHandle in targetHandles)
    {
        UnregisterTarget(targetHandle);
    }
}

void TargetManagerAccessor::EnableCradle(TargetHandle^ targetHandle)
{
    TMAPI_DO(nn::tm::SetTargetDockState(targetHandle->GetHandle(), true));
}

void TargetManagerAccessor::DisableCradle(TargetHandle^ targetHandle)
{
    TMAPI_DO(nn::tm::SetTargetDockState(targetHandle->GetHandle(), false));
}

TargetManagerAccessor::~TargetManagerAccessor()
{
}

TargetManagerAccessor::!TargetManagerAccessor()
{
}

int32_t TargetManagerAccessor::GetHtcsControlPort()
{
    int32_t ret;
    TMAPI_DO(nn::tm::GetHtcsControlPort(&ret));
    return ret;
}

String^ TargetManagerAccessor::GetTargetName(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetName targetName = { 0 };
    const nn::tm::Error error = { nn::tm::GetTargetName(&targetName, targetHandle)};
    String^ name = gcnew String((error == nn::tm::Error_Ok) ? targetName.name : "");
    return name;
}

String^ TargetManagerAccessor::GetTargetTMAVersion(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetAgentVersion agentVersion = { 0 };
    const nn::tm::Error error = { nn::tm::GetTargetAgentVersion( &agentVersion, targetHandle ) };
    String^ tmaVersion = gcnew String( agentVersion.major.ToString() + "." + agentVersion.minor.ToString() + "." + agentVersion.micro.ToString() + "." + agentVersion.revision.ToString() );
    return tmaVersion;
}

String^ TargetManagerAccessor::GetTargetIPAddress(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetIpAddress targetIpAddress = { 0 };
    const nn::tm::Error error = { nn::tm::GetTargetIpAddress( &targetIpAddress, targetHandle ) };
    String^ ipAddress = gcnew String( ( error == nn::tm::Error_Ok ) ? targetIpAddress.ipAddress : "" );
    return ipAddress;
}

String^ TargetManagerAccessor::GetTargetConnectionType(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetConnectionType targetConnectionType = nn::tm::TargetConnectionType_Unknown;
    const nn::tm::Error error = { nn::tm::GetTargetConnectionType( &targetConnectionType, targetHandle ) };

    String^ connectionType = gcnew String( "" );
    if( targetConnectionType == nn::tm::TargetConnectionType_Usb )
    {
        connectionType = "USB";
    }
    else if( targetConnectionType == nn::tm::TargetConnectionType_Ethernet )
    {
        connectionType = "Ethernet";
    }
    else
    {
        connectionType = "Unknown";
    }
    return connectionType;
}

String^ TargetManagerAccessor::GetTargetHardwareType(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetHardwareType targetHardwareType = nn::tm::TargetHardwareType_Unknown;
    const nn::tm::Error error = { nn::tm::GetTargetHardwareType( &targetHardwareType, targetHandle ) };

    String^ hardwareType = gcnew String( "" );
    if( targetHardwareType == nn::tm::TargetHardwareType_Sdev )
    {
        hardwareType = "SDEV";
    }
    else if( targetHardwareType == nn::tm::TargetHardwareType_Edev )
    {
        hardwareType = "EDEV";
    }
    else if( targetHardwareType == nn::tm::TargetHardwareType_Sim )
    {
        hardwareType = "SIM";
    }
    else
    {
        hardwareType = "Unknown";
    }
    return hardwareType;
}

String^ TargetManagerAccessor::GetTargetSerialNumber(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetSerialNumber targetSerialNumber = { 0 };
    const nn::tm::Error error = { nn::tm::GetTargetSerialNumber(&targetSerialNumber, targetHandle) };
    String^ serialNumber = gcnew String((error == nn::tm::Error_Ok ) ? targetSerialNumber.serialNumber : "");
    return serialNumber;
}

String^ TargetManagerAccessor::GetTargetMacAddress( nn::tm::TargetHandle targetHandle )
{
    nn::tm::TargetMacAddress targetMacAddress = { 0 };
    const nn::tm::Error error = { nn::tm::GetTargetMacAddress(&targetMacAddress, targetHandle) };
    String^ macAddress = gcnew String((error == nn::tm::Error_Ok ) ? targetMacAddress.macAddress : "");
    return macAddress;
}

String^ TargetManagerAccessor::GetTargetStatus(nn::tm::TargetHandle targetHandle)
{
    nn::tm::TargetStatus targetStatus = nn::tm::TargetStatus_Unknown;
    const nn::tm::Error error = { nn::tm::GetTargetStatus( &targetStatus, targetHandle ) };

    String^ status = gcnew String( "" );
    if( targetStatus == nn::tm::TargetStatus_Asleep )
    {
        status = "Asleep";
    }
    else if( targetStatus == nn::tm::TargetStatus_Connected )
    {
        status = "Connected";
    }
    else if( targetStatus == nn::tm::TargetStatus_Connecting )
    {
        status = "Connecting";
    }
    else if( targetStatus == nn::tm::TargetStatus_Disconnected )
    {
        status = "Disconnected";
    }
    else if( targetStatus == nn::tm::TargetStatus_Disconnecting )
    {
        status = "Disconnecting";
    }
    else if( targetStatus == nn::tm::TargetStatus_PoweringOn )
    {
        status = "Powering on...";
    }
    else if( targetStatus == nn::tm::TargetStatus_PoweringOff )
    {
        status = "Powering off...";
    }
    else
    {
        status = "Unknown";
    }
    return status;
}

}
}
}
