﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;
using System.Globalization;
using System.Diagnostics;

namespace Nintendo.ControlTarget
{
    public class InvalidEndPointFormat : InvalidUsage
    {
        public InvalidEndPointFormat(string endPoint)
            : base(string.Format("Invalid endpoint format: expected='123.123.123.123:80' or '123.123.123.123', actual='{0}'", endPoint))
        {
        }
    }

    public class NetworkUtility
    {
        public static IPEndPoint Parse(string endPoint, int defaultPort)
        {
            string[] splitted = endPoint.Split(':');
            if (splitted.Length == 1)
            {
                IPAddress address;

                if (!IPAddress.TryParse(endPoint, out address))
                {
                    throw new InvalidEndPointFormat(endPoint);
                }

                return new IPEndPoint(address, defaultPort);
            }
            else if (2 == splitted.Length)
            {
                IPAddress address;
                int port;

                if (!IPAddress.TryParse(splitted[0], out address))
                {
                    throw new InvalidEndPointFormat(endPoint);
                }

                if (!int.TryParse(splitted[splitted.Length - 1], NumberStyles.None, NumberFormatInfo.CurrentInfo, out port))
                {
                    throw new InvalidEndPointFormat(endPoint);
                }

                return new IPEndPoint(address, port);
            }
            else
            {
                throw new InvalidEndPointFormat(endPoint);
            }
        }

        public static Socket CreateConnectedSocket(IPEndPoint endPoint)
        {
            var socket = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);

            try
            {
                RetryUtility.Do(
                    () =>
                    {
                        Trace.WriteLine($"Try to connect: {endPoint}");
                        var result = socket.BeginConnect(endPoint, null, null);
                        if (!result.AsyncWaitHandle.WaitOne(2000, true))
                        {
                            socket.EndConnect(result);
                            throw new TimeoutException();
                        }
                    },
                    e =>
                    {
                        Console.Error.WriteLine($"[ERROR] Failed to connect: {e.Message}");
                    },
                    5,
                    TimeSpan.FromSeconds(1));
            }
            catch (Exception e)
            {
                socket.Dispose();
                throw new Exception($"Failed to connect: {e.Message}");
            }

            return socket;
        }
    }
}
