﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.NetworkInformation;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.Bridge
{
    public class BridgeDetect
    {
        List<NetworkAdapter> adapters = null;
        private int port = 7974;
        private List<BridgeInfo> bridges = null;

        AutoResetEvent evtSearch = null;
        BridgeInfo target = null;
        string key = null;
        SEARCH mode = SEARCH.ALL;

        public enum SEARCH { ALL, BY_SERIAL, BY_MAC_ADDRESS, BY_NAME };

        public BridgeDetect()
        {
            bridges = new List<BridgeInfo>();
            adapters = new List<NetworkAdapter>();

            // すべてのネットワークインターフェイスを取得する
            NetworkInterface[] nis = NetworkInterface.GetAllNetworkInterfaces();

            foreach (NetworkInterface ni in nis)
            {
                // 構成情報、アドレス情報を取得する
                IPInterfaceProperties properties = ni.GetIPProperties();

                if (properties == null || properties.UnicastAddresses.Count == 0)
                {
                    continue;
                }

                foreach (UnicastIPAddressInformation info in properties.UnicastAddresses)
                {
                    // IPv4 のみ対応
                    if (info.Address.AddressFamily != AddressFamily.InterNetwork)
                    {
                        continue;
                    }

                    if (info.IPv4Mask == null)
                    {
                        continue;
                    }

                    if (info.Address.Equals(IPAddress.Loopback))
                    {
                        continue;
                    }

                    NetworkAdapter adapter = new NetworkAdapter(info, port);
                    adapters.Add(adapter);
                }
            }
        }

        private void update(IPEndPoint endPoint, byte[] info)
        {
            if (info.Length < 128)
            {
                return;
            }

            try
            {
                BridgeInfo bridge = new BridgeInfo();
                bridge.SetInfo(endPoint, info);

                switch (mode)
                {
                    case SEARCH.BY_SERIAL:
                        if (bridge.SerialNumber == key)
                        {
                            target = bridge;
                            evtSearch.Set();
                        }
                        break;
                    case SEARCH.BY_MAC_ADDRESS:
                        if (bridge.MacAddress == key)
                        {
                            target = bridge;
                            evtSearch.Set();
                        }
                        break;
                    case SEARCH.BY_NAME:
                        if (bridge.Name == key)
                        {
                            target = bridge;
                            evtSearch.Set();
                        }
                        break;
                }

                bridge.GetInfo(endPoint.Address.ToString(), true);

                foreach (BridgeInfo item in bridges)
                {
                    if (item.MacAddress == bridge.MacAddress)
                    {
                        return;
                    }

                    if (item.SerialNumber == bridge.SerialNumber)
                    {
                        return;
                    }
                }
                bridges.Add(bridge);
            }
            catch (Exception e)
            {
                Console.Write(e);
            }
        }

        bool enumerate(char prefix, int timeout = 8000)
        {
            bool result = true;

            mode = SEARCH.ALL;

            try
            {
                foreach (NetworkAdapter adapter in adapters)
                {
                    adapter.notified += new NotifiedHandler(update);
                    adapter.Start();
                    adapter.LegacyBroadcast(prefix);
                }

                System.Threading.Thread.Sleep(timeout);

                foreach (NetworkAdapter adapter in adapters)
                {
                    adapter.Stop();
                }
            }
            catch (Exception e)
            {
                Console.Write(e);
                result = false;
            }

            return result;
        }

        public bool Enumerate(out List<BridgeInfo> infos, bool detect = false)
        {
            bool result;

            if (detect)
            {
                result = enumerate('?');
            }
            else
            {
                result = enumerate('*');
            }

            infos = bridges;

            return result;
        }

        public bool Enumerate(
            out int targetNum,
            out String[] targetIPs,
            out int[] targetPorts,
            out String[] targetNames,
            out String[] targetMacs,
            out UInt32[] targetVersions,
            out String[] serials,
            int timeout,
            bool detect = false)
        {
            List<BridgeInfo> infos;

            if (Enumerate(out infos, detect))
            {
                targetNum = infos.Count;
                targetIPs = new string[targetNum];
                targetPorts = new int[targetNum];
                targetNames = new string[targetNum];
                targetMacs = new string[targetNum];
                targetVersions = new uint[targetNum];
                serials = new string[targetNum];

                for (int i = 0; i < targetNum; i++)
                {
                    targetIPs[i] = bridges[i].Address;
                    targetPorts[i] = (int)bridges[i].Port;
                    targetNames[i] = bridges[i].Name;
                    targetMacs[i] = bridges[i].MacAddress;
                    targetVersions[i] = bridges[i].Version;
                    serials[i] = bridges[i].SerialNumber;
                }

                return true;
            }
            else
            {
                targetNum = 0;
                targetIPs = null;
                targetPorts = null;
                targetNames = null;
                targetMacs = null;
                targetVersions = null;
                serials = null;

                return false;
            }
        }

        public bool Search(string key, out BridgeInfo info, SEARCH mode, int timeout = 8000)
        {
            bool result = true;

            evtSearch = new AutoResetEvent(false);
            this.key = key;
            this.mode = mode;
            info = null;

            try
            {
                foreach (NetworkAdapter adapter in adapters)
                {
                    adapter.notified += new NotifiedHandler(update);
                    adapter.Start();
                    adapter.LegacyBroadcast('*');
                }

                if (evtSearch.WaitOne(timeout) == false)
                {
                    result = false;
                }

                info = target;

                foreach (NetworkAdapter adapter in adapters)
                {
                    adapter.Stop();
                }
            }
            catch (Exception e)
            {
                Console.Write(e);
                result = false;
            }

            return result;
        }

        public bool Search(string host, out BridgeInfo info, int timeout = 8000)
        {
            info = new BridgeInfo();
            try
            {
                IPAddress address;

                if (!IPAddress.TryParse(host, out address))
                {
                    IPHostEntry hostentry = Dns.GetHostEntry(host);
                    if (hostentry.AddressList.Length > 0)
                    {
                        info.GetInfo(hostentry.AddressList[0].ToString());
                        return info.Valid;
                    }
                    else
                    {
                        return false;
                    }
                }
                else
                {
                    info.GetInfo(host);
                    return info.Valid;
                }
            }
            catch (Exception e)
            {
                Console.Write(e);
                return false;
            }
        }

        public bool Find(string name, string serial, out BridgeInfo info, int timeout = 8000)
        {
            info = new BridgeInfo();

            return false;
        }
    }
}
